/**
 * @fileOverview 拓展分析交互
 * @author huangtonger@aliyun.com
 */
const G6 = require('@antv/g6');
const Util = G6.Util;

function panCanvas(graph, button = 'left') {
  let lastPoint;
  if (button === 'right') {
    graph.behaviourOn('contextmenu', ev => {
      ev.domEvent.preventDefault();
    });
  }
  graph.behaviourOn('mousedown', ev => {
    if (button === 'left' && ev.domEvent.button === 0 || button === 'right' && ev.domEvent.button === 2) {
      lastPoint = {
        x: ev.domX,
        y: ev.domY
      };
    }
  });
  graph.behaviourOn('canvas:mouseenter', () => {
    graph.css({
      cursor: '-webkit-grab'
    });
  });
  graph.behaviourOn('dragstart', ev => {
    graph.css({
      cursor: '-webkit-grabbing'
    });
    if (ev.domEvent.button === button) {
      lastPoint = {
        x: ev.domX,
        y: ev.domY
      };
    }
  });
  graph.behaviourOn('drag', ev => {
    if (lastPoint) {
      graph.translate(ev.domX - lastPoint.x, ev.domY - lastPoint.y);
      lastPoint = {
        x: ev.domX,
        y: ev.domY
      };
    }
  });
  graph.behaviourOn('dragend', () => {
    lastPoint = undefined;
    graph.css({
      cursor: '-webkit-grab'
    });
  });
  graph.behaviourOn('canvas:mouseleave', () => {
    lastPoint = undefined;
  });
}

// 鼠标拖拽平移画布交互
G6.registerBehaviour('panCanvas', panCanvas);

// 鼠标右键平移画布交互
G6.registerBehaviour('rightPanCanvas', graph => {
  panCanvas(graph, 'right');
});

// 鼠标拖拽平移节点交互
G6.registerBehaviour('panNode', graph => {
  let node;
  let dx;
  let dy;
  graph.on('node:dragstart', ev => {
    const {
      item
    } = ev;
    const model = item.getModel();
    node = item;
    dx = model.x - ev.x;
    dy = model.y - ev.y;
    graph.forcePreventAnimate(true);
  });
  graph.on('node:drag', ev => {
    graph.update(node, {
      x: ev.x + dx,
      y: ev.y + dy
    });
  });
  graph.on('node:dragend', () => {
    node = undefined;
    graph.forcePreventAnimate(false);
  });
  graph.on('canvas:mouseleave', () => {
    node = undefined;
    graph.forcePreventAnimate(false);
  });
});

// wheel zoom
G6.registerBehaviour('wheelZoom', graph => {
  let timeout;
  graph.behaviourOn('mousewheel', ev => {
    const { domEvent } = ev;
    domEvent.preventDefault();
  });
  graph.behaviourOn('mousewheel', Util.throttle(update, 16));

  function update(ev) {
    const {
      domEvent
    } = ev;
    const delta = domEvent.wheelDelta;
    const times = 1.05;
    if (Math.abs(delta) > 10) { // 控制灵敏度
      const originScale = graph.getMatrix()[0];
      if (delta > 0) {
        graph.zoom({
          x: ev.x,
          y: ev.y
        }, originScale * times);
      } else {
        graph.zoom({
          x: ev.x,
          y: ev.y
        }, originScale * (1 / times));
      }
    }
    timeout && clearTimeout(timeout);
    timeout = setTimeout(() => {
      timeout = undefined;
    }, 50);
    graph.emit('afterzoom');
  }
});
