/**
 * -------------------------------------------------------------------
 * easy-toggle-state
 * A tiny JavaScript plugin to toggle the state of any HTML element in most of contexts with ease.
 *
 * @author Matthieu Bué <https://twikito.com>
 * @version v1.6.0
 * @link https://twikito.github.io/easy-toggle-state/
 * @license MIT : https://github.com/Twikito/easy-toggle-state/blob/master/LICENSE
 * -------------------------------------------------------------------
 */

(function () {
	'use strict';

	/**
	 * You can change this PREFIX value to prevent conflict with another JS library.
	 * This prefix will be set to all attributes like 'data-[PREFIX]-class'.
	 */
	const PREFIX = "toggle";

	/**
	 * Retrieve a valid HTML attribute string.
	 * @param {string} key - A string to build a html attribute
	 * @returns {string} - A valid html attribute
	 */
	const dataset = key => ["data", PREFIX, key].filter(Boolean).join("-");

	/**
	 * All constants containing HTML attributes string.
	 */
	const CHECKED = "aria-checked",
	      CLASS = dataset("class"),
	      ESCAPE = dataset("escape"),
	      EVENT = dataset("event"),
	      EXPANDED = "aria-expanded",
	      GROUP = dataset("group"),
	      HIDDEN = "aria-hidden",
	      IS_ACTIVE = dataset("is-active"),
	      OUTSIDE = dataset("outside"),
	      RADIO_GROUP = dataset("radio-group"),
	      SELECTED = "aria-selected",
	      TARGET = dataset("target"),
	      TARGET_ALL = dataset("target-all"),
	      TARGET_NEXT = dataset("target-next"),
	      TARGET_ONLY = dataset("target-only"),
	      TARGET_PARENT = dataset("target-parent"),
	      TARGET_PREVIOUS = dataset("target-previous"),
	      TARGET_SELF = dataset("target-self"),
	      TARGET_STATE = dataset("state"),
	      TRIGGER_OFF = dataset("trigger-off");

	/**
	 * Retrieve all trigger elements with a specific attribute, or all nodes in a specific scope.
	 * @param {string} selector - A string that contains a selector
	 * @param {node} [node] - An element in which to make the selection
	 * @returns {array} - An array of elements
	 */
	const $$ = ((selector, node) => {
	  const scope = selector ? `[${selector}]` : "";
	  return node ? [...node.querySelectorAll(scope)] : [...document.querySelectorAll(`[${CLASS}]${scope}`.trim())];
	});

	/**
	 * Aria attributes toggle manager.
	 * @param {node} element - Current element with aria attributes to manage.
	 * @param {json} [config] - List of aria attributes and value to assign.
	 * @returns {undefined}
	 */
	const manageAria = ((element, config = {
		[CHECKED]: element.isToggleActive,
		[EXPANDED]: element.isToggleActive,
		[HIDDEN]: !element.isToggleActive,
		[SELECTED]: element.isToggleActive
	}) => {
		Object.keys(config).forEach(key => element.hasAttribute(key) && element.setAttribute(key, config[key]));
	});

	/**
	 * Retrieve all active elements of a group.
	 * @param {node} element - An element of a group
	 * @returns {array} - An array of active elements of a group
	 */
	const retrieveGroupActiveElement = (element => {
	  const type = element.hasAttribute(GROUP) ? GROUP : RADIO_GROUP;
	  return $$(`${type}="${element.getAttribute(type)}"`).filter(groupElement => groupElement.isToggleActive);
	});

	/**
	 * Test a targets list.
	 * @param {string} selector - The selector corresponding to the targets list
	 * @param {nodeList} targetList - A target elements list
	 * @returns {nodeList} - The targets list
	 */
	const testTargets = (selector, targetList) => {

		/** Test if there's no match for a selector */
		if (targetList.length === 0) {
			console.warn(`There's no match for the selector '${selector}' for this trigger`);
		}

		/** Test if there's more than one match for an ID selector */
		const matches = selector.match(/#\w+/gi);
		if (matches) {
			matches.forEach(match => {
				const result = [...targetList].filter(target => target.id === match.slice(1));
				if (result.length > 1) {
					console.warn(`There's ${result.length} matches for the selector '${match}' for this trigger`);
				}
			});
		}

		return targetList;
	};

	/**
	 * Retrieve all targets of a trigger element, depending of its target attribute.
	 * @param {node} element - A trigger element
	 * @returns {nodeList} - All targets of a trigger element
	 */
	const retrieveTargets = (element => {
		if (element.hasAttribute(TARGET) || element.hasAttribute(TARGET_ALL)) {
			const selector = element.getAttribute(TARGET) || element.getAttribute(TARGET_ALL);
			return testTargets(selector, document.querySelectorAll(selector));
		}

		if (element.hasAttribute(TARGET_PARENT)) {
			const selector = element.getAttribute(TARGET_PARENT);
			return testTargets(selector, element.parentElement.querySelectorAll(selector));
		}

		if (element.hasAttribute(TARGET_SELF)) {
			const selector = element.getAttribute(TARGET_SELF);
			return testTargets(selector, element.querySelectorAll(selector));
		}

		if (element.hasAttribute(TARGET_PREVIOUS)) {
			return testTargets("previous", [element.previousElementSibling].filter(Boolean));
		}

		if (element.hasAttribute(TARGET_NEXT)) {
			return testTargets("next", [element.nextElementSibling].filter(Boolean));
		}

		return [];
	});

	/**
	 * Toggle off all elements width 'data-toggle-outside' attribute
	 * when reproducing specified or click event outside itself or its targets.
	 * @param {event} event - Event triggered on document
	 * @returns {undefined}
	 */
	const documentEventHandler = event => {
		const target = event.target;
		if (!target.closest("[" + TARGET_STATE + '="true"]')) {
			$$(OUTSIDE).forEach(element => {
				if (element !== target && element.isToggleActive) {
					(element.hasAttribute(GROUP) || element.hasAttribute(RADIO_GROUP) ? manageGroup : manageToggle)(element);
				}
			});
			if (target.hasAttribute(OUTSIDE) && target.isToggleActive) {
				document.addEventListener(target.getAttribute(EVENT) || "click", documentEventHandler, false);
			}
		}
	};

	/**
	 * Manage click on elements with 'data-trigger-off' attribue.
	 * @param {event} event - Event triggered on element with 'trigger-off' attribute
	 * @returns {undefined}
	 */
	const triggerOffHandler = event => {
		manageToggle(event.target.targetElement);
	};

	/**
	 * Manage attributes and events of target elements.
	 * @param {node} targetElement - An element targeted by the trigger element
	 * @param {node} triggerElement - The trigger element
	 * @returns {undefined}
	 */
	const manageTarget = (targetElement, triggerElement) => {
		targetElement.isToggleActive = !targetElement.isToggleActive;
		manageAria(targetElement);

		if (triggerElement.hasAttribute(OUTSIDE)) {
			targetElement.setAttribute(TARGET_STATE, triggerElement.isToggleActive);
		}

		const triggerOffList = $$(TRIGGER_OFF, targetElement);
		if (triggerOffList.length > 0) {
			if (triggerElement.isToggleActive) {
				triggerOffList.forEach(triggerOff => {
					triggerOff.targetElement = triggerElement;
					triggerOff.addEventListener("click", triggerOffHandler, false);
				});
			} else {
				triggerOffList.forEach(triggerOff => {
					triggerOff.removeEventListener("click", triggerOffHandler, false);
				});
			}
		}
	};

	/**
	 * Toggle class and aria on trigger and target elements.
	 * @param {node} element - The element to toggle state and attributes
	 * @returns {undefined}
	 */
	const manageToggle = element => {
		const className = element.getAttribute(CLASS) || "is-active";
		element.isToggleActive = !element.isToggleActive;
		manageAria(element);

		if (!element.hasAttribute(TARGET_ONLY)) {
			element.classList.toggle(className);
		}

		const targetElements = retrieveTargets(element);
		for (let i = 0; i < targetElements.length; i++) {
			targetElements[i].classList.toggle(className);
			manageTarget(targetElements[i], element);
		}

		manageTriggerOutside(element);
	};

	/**
	 * Manage event ouside trigger or target elements.
	 * @param {node} element - The element to toggle when 'click' or custom event is triggered on document
	 * @returns {undefined}
	 */
	const manageTriggerOutside = element => {
		if (element.hasAttribute(OUTSIDE)) {
			if (element.hasAttribute(RADIO_GROUP)) {
				console.warn(`You can't use '${OUTSIDE}' on a radio grouped trigger`);
			} else {
				if (element.isToggleActive) {
					document.addEventListener(element.getAttribute(EVENT) || "click", documentEventHandler, false);
				} else {
					document.removeEventListener(element.getAttribute(EVENT) || "click", documentEventHandler, false);
				}
			}
		}
	};

	/**
	 * Toggle elements of a same group.
	 * @param {node} element - The element to test if it's in a group
	 * @returns {undefined}
	 */
	const manageGroup = element => {
		const groupActiveElements = retrieveGroupActiveElement(element);
		if (groupActiveElements.length > 0) {
			if (groupActiveElements.indexOf(element) === -1) {
				groupActiveElements.forEach(manageToggle);
				manageToggle(element);
			}
			if (groupActiveElements.indexOf(element) !== -1 && !element.hasAttribute(RADIO_GROUP)) {
				manageToggle(element);
			}
		} else {
			manageToggle(element);
		}
	};

	/**
	 * Toggle elements set to be active by default.
	 * @param {node} element - The element to activate on page load
	 * @returns {undefined}
	 */
	const manageActiveByDefault = element => {
		const className = element.getAttribute(CLASS) || "is-active";
		element.isToggleActive = true;
		manageAria(element, {
			[CHECKED]: true,
			[EXPANDED]: true,
			[HIDDEN]: false,
			[SELECTED]: true
		});

		if (!element.hasAttribute(TARGET_ONLY) && !element.classList.contains(className)) {
			element.classList.add(className);
		}

		const targetElements = retrieveTargets(element);
		for (let i = 0; i < targetElements.length; i++) {
			if (!targetElements[i].classList.contains(className)) {
				targetElements[i].classList.add(className);
			}
			manageTarget(targetElements[i], element);
		}

		manageTriggerOutside(element);
	};

	/**
	 * Initialization.
	 * @returns {undefined}
	 */
	const init = (() => {

		/** Test if there's some trigger */
		if ($$().length === 0) {
			return console.warn(`Easy Toggle State is not used: there's no trigger to initialize.`);
		}

		/** Active by default management. */
		$$(IS_ACTIVE).forEach(trigger => {
			if (trigger.hasAttribute(GROUP) || trigger.hasAttribute(RADIO_GROUP)) {
				if (retrieveGroupActiveElement(trigger).length > 0) {
					console.warn(`Toggle group '${trigger.getAttribute(GROUP) || trigger.getAttribute(RADIO_GROUP)}' must not have more than one trigger with '${IS_ACTIVE}'`);
				} else {
					manageActiveByDefault(trigger);
				}
			} else {
				manageActiveByDefault(trigger);
			}
		});

		/** Set specified or click event on each trigger element. */
		$$().forEach(trigger => {
			trigger.addEventListener(trigger.getAttribute(EVENT) || "click", event => {
				event.preventDefault();
				(trigger.hasAttribute(GROUP) || trigger.hasAttribute(RADIO_GROUP) ? manageGroup : manageToggle)(trigger);
			}, false);
		});

		/** Escape key management. */
		const triggerEscElements = $$(ESCAPE);
		if (triggerEscElements.length > 0) {
			document.addEventListener("keyup", event => {
				event = event || window.event;
				if (event.key === "Escape" || event.key === "Esc") {
					triggerEscElements.forEach(trigger => {
						if (trigger.isToggleActive) {
							if (trigger.hasAttribute(RADIO_GROUP)) {
								console.warn(`You can't use '${ESCAPE}' on a radio grouped trigger`);
							} else {
								(trigger.hasAttribute(GROUP) ? manageGroup : manageToggle)(trigger);
							}
						}
					});
				}
			}, false);
		}
	});

	/* eslint no-unused-vars: "off" */

	const onLoad = () => {
		init();
		document.removeEventListener("DOMContentLoaded", onLoad);
	};

	document.addEventListener("DOMContentLoaded", onLoad);
	window.initEasyToggleState = init;

}());
