﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Reflection;
using System.Web;

namespace DynamicSearchesExample.Models
{

    /// <summary>
    /// Enumerates all types of property value comparisons
    /// </summary>
    public enum PropertyValueComparison
    {
        Equals = 0,
        NotEquals = 1,
        IsGreaterThan = 2,
        IsGreaterOrEquals = 3,
        IsLowerThan = 4,
        IsLowerThanOrEquals = 5,
        IsNull = 6,
        IsNotNull = 7,
        StringContains = 8,
        StringDoesNotContain = 9,
        StringStartsWith = 10,
        StringEndsWith = 11
    }

    /// <summary>
    /// Represents a <see cref="QueryFilter"/> used to filter a query based on a property value comparison
    /// </summary>
    public class WhereFilter
       : QueryFilter
    {

        private static readonly MethodInfo ToStringMethod = typeof(object).GetMethods().Single(m => m.Name == "ToString");
        private static readonly MethodInfo StringStartsWithMethod = typeof(string).GetMethods().Single(m => m.Name == "StartsWith" && m.GetParameters().Length == 1);
        private static readonly MethodInfo StringEndsWithMethod = typeof(string).GetMethods().Single(m => m.Name == "EndsWith" && m.GetParameters().Length == 1);
        private static readonly MethodInfo StringContainsMethod = typeof(string).GetMethods().Single(m => m.Name == "Contains");
        private static readonly MethodInfo StringToLowerMethod = typeof(string).GetMethods().Single(m => m.Name == "ToLower" && m.GetParameters().Length == 0);
        private static readonly MethodInfo WhereMethod = typeof(Queryable).GetMethods().Where(m => m.Name == "Where" && m.GetParameters().Length == 2).First();

        /// <summary>
        /// Gets/sets the raw property path
        /// </summary>
        public string PropertyPath { get; set; }

        /// <summary>
        /// Gets/sets the <see cref="WhereFilter"/>'s <see cref="PropertyValueComparison"/>
        /// </summary>
        public PropertyValueComparison ValueComparison { get; set; }

        /// <summary>
        /// Gets/sets the value to compare
        /// </summary>
        public string Value { get; set; }

        /// <summary>
        /// Gets/sets a boolean indicating whether or not the comparison should be case-sensitive
        /// </summary>
        public bool CaseSensitive { get; set; }

        /// <summary>
        /// Filters the specified query
        /// </summary>
        /// <param name="entityType">The type of the query</param>
        /// <param name="query">The query to filter</param>
        /// <returns>The filtered query</returns>
        public override IQueryable Filter(Type entityType, IQueryable query)
        {
            ParameterExpression parameterExpression;
            PropertyPath propertyPath;
            MemberExpression getPropertyExpression;
            ConstantExpression constantExpression;
            BinaryExpression binaryExpression;
            Expression stringExpression;
            MethodCallExpression callExpression, filterExpression;
            MethodInfo whereMethod;
            object constant;
            string value;
            parameterExpression = Expression.Parameter(entityType, "param");
            if (!DynamicSearchesExample.PropertyPath.TryParse(this.PropertyPath, out propertyPath))
            {
                throw new Exception("Failed to parse the specified value '" + this.PropertyPath + "' into a " + nameof(DynamicSearchesExample.PropertyPath));
            }
            getPropertyExpression = propertyPath.ToExpression(parameterExpression);
            constant = null;
            stringExpression = null;
            if ((int)this.ValueComparison < 8)
            {
                //Normal value comparison -> attempt to convert this.Value into expected type
                if (!ParserHelper.TryParse(getPropertyExpression.Type, this.Value, out constant))
                {
                    throw new Exception("Failed to parse the specified value '" + this.Value + "' into expected type '" + getPropertyExpression.Type.FullName + "'");
                }
            }
            else
            {
                if(getPropertyExpression.Type == typeof(string))
                {
                    stringExpression = getPropertyExpression;
                }
                else
                {
                    stringExpression = Expression.Call(getPropertyExpression, ToStringMethod);
                }
            }
            switch (this.ValueComparison)
            {
                case PropertyValueComparison.Equals:
                    constantExpression = Expression.Constant(constant);
                    if (getPropertyExpression.Type.IsNullable()
                        && getPropertyExpression.Type != typeof(string))
                    {
                        getPropertyExpression = Expression.MakeMemberAccess(getPropertyExpression, typeof(Nullable<>).MakeGenericType(constantExpression.Type).GetProperty("Value"));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.Equal, getPropertyExpression, constantExpression);
                    break;
                case PropertyValueComparison.IsNull:
                    constantExpression = Expression.Constant(null);
                    binaryExpression = Expression.MakeBinary(ExpressionType.Equal, getPropertyExpression, constantExpression);
                    break;
                case PropertyValueComparison.NotEquals:
                    constantExpression = Expression.Constant(constant);
                    if (getPropertyExpression.Type.IsNullable()
                         && getPropertyExpression.Type != typeof(string))
                    {
                        getPropertyExpression = Expression.MakeMemberAccess(getPropertyExpression, typeof(Nullable<>).MakeGenericType(constantExpression.Type).GetProperty("Value"));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.NotEqual, getPropertyExpression, constantExpression);
                    break;
                case PropertyValueComparison.IsNotNull:
                    constantExpression = Expression.Constant(null);
                    binaryExpression = Expression.MakeBinary(ExpressionType.NotEqual, getPropertyExpression, constantExpression);
                    break;
                case PropertyValueComparison.IsGreaterThan:
                    constantExpression = Expression.Constant(constant);
                    if (getPropertyExpression.Type.IsNullable()
                         && getPropertyExpression.Type != typeof(string))
                    {
                        getPropertyExpression = Expression.MakeMemberAccess(getPropertyExpression, typeof(Nullable<>).MakeGenericType(constantExpression.Type).GetProperty("Value"));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.GreaterThan, getPropertyExpression, constantExpression);
                    break;
                case PropertyValueComparison.IsGreaterOrEquals:
                    constantExpression = Expression.Constant(constant);
                    if (getPropertyExpression.Type.IsNullable()
                         && getPropertyExpression.Type != typeof(string))
                    {
                        getPropertyExpression = Expression.MakeMemberAccess(getPropertyExpression, typeof(Nullable<>).MakeGenericType(constantExpression.Type).GetProperty("Value"));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.GreaterThanOrEqual, getPropertyExpression, constantExpression);
                    break;
                case PropertyValueComparison.IsLowerThan:
                    constantExpression = Expression.Constant(constant);
                    if (getPropertyExpression.Type.IsNullable()
                         && getPropertyExpression.Type != typeof(string))
                    {
                        getPropertyExpression = Expression.MakeMemberAccess(getPropertyExpression, typeof(Nullable<>).MakeGenericType(constantExpression.Type).GetProperty("Value"));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.LessThan, getPropertyExpression, constantExpression);
                    break;
                case PropertyValueComparison.IsLowerThanOrEquals:
                    constantExpression = Expression.Constant(constant);
                    if (getPropertyExpression.Type.IsNullable()
                         && getPropertyExpression.Type != typeof(string))
                    {
                        getPropertyExpression = Expression.MakeMemberAccess(getPropertyExpression, typeof(Nullable<>).MakeGenericType(constantExpression.Type).GetProperty("Value"));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.LessThanOrEqual, getPropertyExpression, constantExpression);
                    break;
                case PropertyValueComparison.StringContains:
                    constantExpression = Expression.Constant(true);
                    if (this.CaseSensitive)
                    {
                        value = this.Value;
                        callExpression = Expression.Call(stringExpression, WhereFilter.StringContainsMethod, Expression.Constant(value));
                    }
                    else
                    {
                        value = this.Value.ToLower();
                        callExpression = Expression.Call(stringExpression, WhereFilter.StringToLowerMethod);
                        callExpression = Expression.Call(callExpression, WhereFilter.StringContainsMethod, Expression.Constant(value));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.Equal, callExpression, constantExpression);
                    break;
                case PropertyValueComparison.StringDoesNotContain:
                    constantExpression = Expression.Constant(false);
                    if (this.CaseSensitive)
                    {
                        value = this.Value;
                        callExpression = Expression.Call(stringExpression, WhereFilter.StringContainsMethod, Expression.Constant(value));
                    }
                    else
                    {
                        value = this.Value.ToLower();
                        callExpression = Expression.Call(stringExpression, WhereFilter.StringToLowerMethod);
                        callExpression = Expression.Call(callExpression, WhereFilter.StringContainsMethod, Expression.Constant(value));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.Equal, callExpression, constantExpression);
                    break;
                case PropertyValueComparison.StringStartsWith:
                    constantExpression = Expression.Constant(true);
                    if (this.CaseSensitive)
                    {
                        value = this.Value;
                        callExpression = Expression.Call(stringExpression, WhereFilter.StringStartsWithMethod, Expression.Constant(value));
                    }
                    else
                    {
                        value = this.Value.ToLower();
                        callExpression = Expression.Call(stringExpression, WhereFilter.StringToLowerMethod);
                        callExpression = Expression.Call(callExpression, WhereFilter.StringStartsWithMethod, Expression.Constant(value));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.Equal, callExpression, constantExpression);
                    break;
                case PropertyValueComparison.StringEndsWith:
                    constantExpression = Expression.Constant(true);
                    if (this.CaseSensitive)
                    {
                        value = this.Value;
                        callExpression = Expression.Call(stringExpression, WhereFilter.StringEndsWithMethod, Expression.Constant(value));
                    }
                    else
                    {
                        value = this.Value.ToLower();
                        callExpression = Expression.Call(stringExpression, WhereFilter.StringToLowerMethod);
                        callExpression = Expression.Call(callExpression, WhereFilter.StringEndsWithMethod, Expression.Constant(value));
                    }
                    binaryExpression = Expression.MakeBinary(ExpressionType.Equal, callExpression, constantExpression);
                    break;
                default:
                    throw new NotSupportedException("The specified " + nameof(PropertyValueComparison) + " '" + this.ValueComparison.ToString() + "' is not supported");
            }
            whereMethod = WhereFilter.WhereMethod.MakeGenericMethod(entityType);
            filterExpression = Expression.Call(whereMethod, new Expression[] { query.Expression, Expression.Lambda(binaryExpression, parameterExpression) });
            return query.Provider.CreateQuery(filterExpression);
        }

    }

}