﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Reflection;
using System.Web;

namespace DynamicSearchesExample.Models
{

    /// <summary>
    /// Represents a <see cref="QueryFilter"/> used to chain OrderBy calls
    /// </summary>
    public class ThenByFilter
        : QueryFilter
    {

        private static readonly MethodInfo ThenByMethod = typeof(Queryable).GetMethods().Single(m => m.Name == "ThenBy" && m.GetParameters().Length == 2);
        private static readonly MethodInfo ThenByDescendingMethod = typeof(Queryable).GetMethods().Single(m => m.Name == "ThenByDescending" && m.GetParameters().Length == 2);

        /// <summary>
        /// Gets/sets a boolean indicating whether the OrderBy is descending or ascending
        /// </summary>
        public bool Descending { get; set; }

        /// <summary>
        /// Gets/sets the raw property path
        /// </summary>
        public string PropertyPath { get; set; }

        /// <summary>
        /// Filters the specified query
        /// </summary>
        /// <param name="entityType">The type of the query</param>
        /// <param name="query">The query to filter</param>
        /// <returns>The filtered query</returns>
        public override IQueryable Filter(Type entityType, IQueryable query)
        {
            PropertyPath propertyPath;
            ParameterExpression parameterExpression;
            MemberExpression getPropertyExpression;
            LambdaExpression lambdaExpression;
            MethodInfo orderByMethod;
            MethodCallExpression filterExpression;
            parameterExpression = Expression.Parameter(entityType, "param");
            if (!DynamicSearchesExample.PropertyPath.TryParse(this.PropertyPath, out propertyPath))
            {
                throw new Exception(string.Format("Failed to parse the specified value '{0}' into a {1}", this.PropertyPath, nameof(DynamicSearchesExample.PropertyPath)));
            }
            getPropertyExpression = propertyPath.ToExpression(parameterExpression);
            lambdaExpression = Expression.Lambda(getPropertyExpression, parameterExpression);
            if (this.Descending)
            {
                orderByMethod = ThenByFilter.ThenByDescendingMethod.MakeGenericMethod(entityType, getPropertyExpression.Type);
            }
            else
            {
                orderByMethod = ThenByFilter.ThenByMethod.MakeGenericMethod(entityType, getPropertyExpression.Type);
            }
            filterExpression = Expression.Call(orderByMethod, query.Expression, Expression.Quote(lambdaExpression));
            return query.Provider.CreateQuery(filterExpression);
        }

    }

}