﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Data.Entity;
using System.Linq;
using System.Reflection;
using System.Web;

namespace DynamicSearchesExample.Models
{

    /// <summary>
    /// The view model of a filterable query
    /// </summary>
    /// <typeparam name="T">The type of the query</typeparam>
    public class QueryViewModel<T>
        where T : class
    {

        private static MethodInfo CountMethod = typeof(Queryable).GetMethods().Single(m => m.Name == "Count" && m.GetParameters().Count() == 1);
        private static MethodInfo SkipMethod = typeof(Queryable).GetMethods().Single(m => m.Name == "Skip" && m.GetParameters().Count() == 2);
        private static MethodInfo TakeMethod = typeof(Queryable).GetMethods().Single(m => m.Name == "Take" && m.GetParameters().Count() == 2);
        private static MethodInfo ToListMethod = typeof(Enumerable).GetMethod("ToList");

        /// <summary>
        /// Gets/sets the maximum results per page
        /// </summary>
        public int ResultsPerPage { get; set; }

        /// <summary>
        /// Gets/sets the current page index
        /// </summary>
        public int PageIndex { get; set; }

        /// <summary>
        /// Gets/sets the current page count
        /// </summary>
        public int PageCount { get; set; }

        /// <summary>
        /// Gets/sets the results of the query
        /// </summary>
        public IEnumerable<T> Results { get; set; }

        /// <summary>
        /// Gets/sets the <see cref="QueryFilter"/>s associated with the query
        /// </summary>
        public IEnumerable<QueryFilter> Filters { get; set; }

        /// <summary>
        /// Executes the query represented by the <see cref="QueryViewModel{T}"/> in the specified <see cref="DbContext"/>
        /// </summary>
        /// <param name="context">The <see cref="DbContext"/> to execute the query into</param>
        public void ExecuteQuery(DbContext context)
        {
            IQueryable query;
            MethodInfo countMethod, skipMethod, takeMethod, toListMethod;
            int pageCount, remainder;
            //Create the query
            query = context.Set<T>();
            if(query == null)
            {
                throw new NullReferenceException(string.Format("Failed to find a {0} of the specified type '{1}'", nameof(DbSet), typeof(T).Name));
            }
            if(this.Filters != null)
            {
                //Apply each filter to the query
                foreach (QueryFilter queryFilter in this.Filters)
                {
                    query = queryFilter.Filter(query.ElementType, query);
                }
            }
            //If we dont do the following, which is a nasty trick, an exception will be thrown when attempting the following Skip() call
            if (!typeof(IOrderedQueryable).IsAssignableFrom(query.Expression.Type)
                || this.Filters == null)
            {
                query = new OrderByFilter() { PropertyPath = query.ElementType.GetProperties().First().Name }.Filter(query.ElementType, query);
            }
            countMethod = CountMethod.MakeGenericMethod(query.ElementType);
            pageCount = Math.DivRem((int)countMethod.Invoke(null, new object[] { query }), this.ResultsPerPage, out remainder);
            if (remainder != 0)
            {
                pageCount++;
            }
            this.PageCount = pageCount;
            skipMethod = SkipMethod.MakeGenericMethod(query.ElementType);
            query = (IQueryable)skipMethod.Invoke(null, new object[] { query, this.ResultsPerPage * this.PageIndex });
            takeMethod = TakeMethod.MakeGenericMethod(query.ElementType);
            query = (IQueryable)takeMethod.Invoke(null, new object[] { query, this.ResultsPerPage });
            toListMethod = ToListMethod.MakeGenericMethod(query.ElementType);
            this.Results = (IEnumerable<T>)toListMethod.Invoke(null, new object[] { query });
        }

    }

}