﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Reflection;
using System.Web;

namespace DynamicSearchesExample.Models
{

    /// <summary>
    /// Represents a <see cref="QueryFilter"/> used to order the results of a query
    /// </summary>
    public class OrderByFilter
        : QueryFilter
    {

        private static readonly MethodInfo OrderByMethod = typeof(Queryable).GetMethods().Single(m => m.Name == "OrderBy" && m.GetParameters().Length == 2);
        private static readonly MethodInfo OrderByDescendingMethod = typeof(Queryable).GetMethods().Single(m => m.Name == "OrderByDescending" && m.GetParameters().Length == 2);

        /// <summary>
        /// Gets/sets a boolean indicating whether the OrderBy is descending or ascending
        /// </summary>
        public bool Descending { get; set; }

        /// <summary>
        /// Gets/sets the raw property path
        /// </summary>
        public string PropertyPath { get; set; }

        /// <summary>
        /// Filters the specified query
        /// </summary>
        /// <param name="entityType">The type of the query</param>
        /// <param name="query">The query to filter</param>
        /// <returns>The filtered query</returns>
        public override IQueryable Filter(Type entityType, IQueryable query)
        {
            PropertyPath propertyPath;
            ParameterExpression parameterExpression;
            MemberExpression getPropertyExpression;
            LambdaExpression lambdaExpression;
            MethodInfo orderByMethod;
            MethodCallExpression filterExpression;
            //Creates the parameter expression
            parameterExpression = Expression.Parameter(entityType, "param");
            //Attempts to parse the PropertyPath
            if (!DynamicSearchesExample.PropertyPath.TryParse(this.PropertyPath, out propertyPath))
            {
                throw new Exception(string.Format("Failed to parse the specified value '{0}' into a {1}", this.PropertyPath, nameof(DynamicSearchesExample.PropertyPath)));
            }
            //Creates the expression to get the value returned by the targeted property (ex: 'param.Property1.Property2')
            getPropertyExpression = propertyPath.ToExpression(parameterExpression);
            //Creates the lambda (ex: '(param) -> param.Property1.Property2')
            lambdaExpression = Expression.Lambda(getPropertyExpression, parameterExpression);
            //Check whether or not the OrderBy is descending
            if (this.Descending)
            {
                //The OrderByDescending method
                orderByMethod = OrderByFilter.OrderByDescendingMethod.MakeGenericMethod(entityType, getPropertyExpression.Type);
            }
            else
            {
                //The OrderBy method
                orderByMethod = OrderByFilter.OrderByMethod.MakeGenericMethod(entityType, getPropertyExpression.Type);
            }
            //Create the filter expression (ex: 'query.OrderBy((param) -> param.Property1.Property2)')
            filterExpression = Expression.Call(orderByMethod, query.Expression, Expression.Quote(lambdaExpression));
            return query.Provider.CreateQuery(filterExpression);
        }

    }

}