﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Reflection;
using System.Web;

namespace DynamicSearchesExample
{

    /// <summary>
    /// Defines and exposes static members to help with the parsing of data
    /// </summary>
    public static class ParserHelper
    {

        /// <summary>
        /// Attempts to parse the input into the specified type
        /// </summary>
        /// <param name="expectedType">The expected type of the parsed input</param>
        /// <param name="input">The string to parse</param>
        /// <param name="result">The resulting object, if any</param>
        /// <returns>A boolean indicating whether or not the specified input could be parsed</returns>
        public static bool TryParse(Type expectedType, string input, out object result)
        {
            TypeConverterAttribute converterAttribute;
            Type converterType;
            TypeConverter converter;
            if (expectedType == typeof(string) || expectedType == typeof(char))
            {
                result = input;
                return true;
            }
            if (expectedType.IsSystemValueType())
            {
                return ParserHelper.TryParseValueType(expectedType, input, out result);
            }
            converterAttribute = expectedType.GetCustomAttribute<TypeConverterAttribute>();
            if (converterAttribute == null)
            {
                result = null;
                return false;
            }
            converterType = Type.GetType(converterAttribute.ConverterTypeName);
            converter = (TypeConverter)Activator.CreateInstance(converterType);
            if (!converter.CanConvertFrom(typeof(string)))
            {
                result = null;
                return false;
            }
            result = converter.ConvertFromString(input);
            return true;
        }

        /// <summary>
        /// Attempts to parse the input into the specified type
        /// </summary>
        /// <typeparam name="TResult">The expected type of the parsed input</typeparam>
        /// <param name="input">The string to parse</param>
        /// <param name="result">The resulting object, if any</param>
        /// <returns>A boolean indicating whether or not the specified input could be parsed</returns>
        public static bool TryParse<TResult>(string input, out TResult result)
        {
            object value;
            if (ParserHelper.TryParse(typeof(TResult), input, out value))
            {
                result = (TResult)value;
                return true;
            }
            else
            {
                result = default(TResult);
                return false;
            }
        }

        /// <summary>
        /// Attempts to parse the input into specified value type
        /// </summary>
        /// <param name="expectedType">The expected type of the parsed input</param>
        /// <param name="input">The string to parse</param>
        /// <param name="result">The resulting object, if any</param>
        /// <returns>A boolean indicating whether or not the specified input could be parsed</returns>
        private static bool TryParseValueType(Type expectedType, string input, out object result)
        {
            bool boolean;
            byte b;
            int integer;
            decimal dec;
            double dbl;
            long lng;
            float flt;
            Guid guid;
            DateTime dateTime;
            Version version;
            if (expectedType.IsNullable())
            {
                expectedType = expectedType.GetGenericArguments().First();
            }
            if (!expectedType.IsSystemValueType())
            {
                result = null;
                return false;
            }
            if (expectedType.IsEnum)
            {
                result = Enum.Parse(expectedType, input);
                return true;
            }
            if (expectedType == typeof(bool))
            {
                if (bool.TryParse(input, out boolean))
                {
                    result = boolean;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            if (expectedType == typeof(byte))
            {
                if (byte.TryParse(input, out b))
                {
                    result = b;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            if (expectedType == typeof(int))
            {
                if (int.TryParse(input, out integer))
                {
                    result = integer;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            if (expectedType == typeof(decimal))
            {
                if (decimal.TryParse(input, out dec))
                {
                    result = dec;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            if (expectedType == typeof(double))
            {
                if (double.TryParse(input, out dbl))
                {
                    result = dbl;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            if (expectedType == typeof(long))
            {
                if (long.TryParse(input, out lng))
                {
                    result = lng;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            if (expectedType == typeof(float))
            {
                if (float.TryParse(input, out flt))
                {
                    result = flt;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            if (expectedType == typeof(Guid))
            {
                if (Guid.TryParse(input, out guid))
                {
                    result = guid;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            if (expectedType == typeof(DateTime))
            {
                if (DateTime.TryParse(input, out dateTime))
                {
                    result = dateTime;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            if (expectedType == typeof(Version))
            {
                if (Version.TryParse(input, out version))
                {
                    result = version;
                    return true;
                }
                else
                {
                    result = null;
                    return false;
                }
            }
            result = null;
            return false;
        }

        /// <summary>
        /// Parses the input into the specified type
        /// </summary>
        /// <param name="expectedType">The expected type of the parsed input</param>
        /// <param name="input">The string to parse</param>
        /// <returns>The resulting object, if any</returns>
        public static object Parse(Type expectedType, string input)
        {
            object result;
            if (!ParserHelper.TryParse(expectedType, input, out result))
            {
                throw new FormatException("Failed to parse the specified input '" + input + "' into expected type '" + expectedType.FullName + "'");
            }
            return result;
        }

        /// <summary>
        /// Parses the input into the specified type
        /// </summary>
        /// <typeparam name="TResult">The expected type of the parsed input</typeparam>
        /// <param name="input">The string to parse</param>
        /// <returns>The resulting object, if any</returns>
        public static TResult Parse<TResult>(string input)
        {
            object result;
            result = ParserHelper.Parse(typeof(TResult), input);
            if (result == null)
            {
                return default(TResult);
            }
            return (TResult)result;
        }

        /// <summary>
        /// Stringifies the specified value
        /// </summary>
        /// <param name="value">The value to stringify</param>
        /// <returns>The stringified representation of the specified value</returns>
        public static string Stringify(object value)
        {
            Type valueType;
            if (value == null)
            {
                return string.Empty;
            }
            valueType = value.GetType();
            if (!valueType.IsSystemValueType())
            {
                throw new ArgumentException("Invalid argument. View inner exception for further details.", "value", new NotSupportedException("The ParserHelper can only stringify system value types plus string, DateTime, Timespan and Guid values"));
            }
            return value.ToString();
        }

    }

}