﻿using System;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Identity.EntityFrameworkCore;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Logging;
using webApp.Data;
using webApp.Models;
using webApp.Services;

// https://docs.microsoft.com/en-us/aspnet/core/mvc/controllers/dependency-injection
// http://ardalis.com/new-is-glue

namespace webApp
{
  public class Startup
  {
    public Startup(IHostingEnvironment env)
    {
      var builder = new ConfigurationBuilder()
           .SetBasePath(env.ContentRootPath)
          .AddJsonFile("appsettings.json", optional: true, reloadOnChange: true)
         .AddJsonFile($"appsettings.{env.EnvironmentName}.json", optional: true);

      builder.AddEnvironmentVariables();
      Configuration = builder.Build();
    }

    public IConfigurationRoot Configuration { get; }

    // This method gets called by the runtime. Use this method to add services to the container.
    public void ConfigureServices(IServiceCollection services)
    {
      var connectionString = Configuration.GetConnectionString("DefaultConnection");

      // Add framework services.
      services.AddDbContext<ApplicationDbContext>(options => options.UseMySql(connectionString));

      services.AddIdentity<ApplicationUser, IdentityRole>()
          .AddEntityFrameworkStores<ApplicationDbContext>()
          .AddDefaultTokenProviders();

      services.AddMvc();


      services.AddTransient(typeof(IRepository<>), typeof(Repository<>));


      //services.AddTransient<IHorseService, HorseService>();

      services.AddDbContext<GriffioenContext>(options => options.UseMySql(connectionString));

      // Read email settings
      services.Configure<EmailConfig>(Configuration.GetSection("Email"));

      // Inject email service 
      services.AddTransient<IEmailService, EmailService>();
    }

    // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
    public void Configure(IApplicationBuilder app, IHostingEnvironment env, ILoggerFactory loggerFactory)
    {
      loggerFactory.AddConsole(Configuration.GetSection("Logging"));
      loggerFactory.AddDebug();

      if (env.IsDevelopment())
      {
        app.UseDeveloperExceptionPage();
        app.UseDatabaseErrorPage();
        app.UseBrowserLink();
      }
      else
      {
        app.UseExceptionHandler("/Home/Error");

        app.Use(async (context, next) =>
        {
          if (context.Request.IsHttps)
            await next();
          else
          {
            var httpsUrl = "https://" + context.Request.Host + context.Request.Path;
            context.Response.Redirect(httpsUrl);
          }
        });
      }

      app.UseStaticFiles();
      app.UseIdentity();

      app.UseMvc(routes => { routes.MapRoute(name: "default", template: "{controller=Home}/{action=Index}/{id?}"); });
    }
  }
}
