﻿using MailKit.Net.Smtp;
using MailKit.Security;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Options;
using MimeKit;
using MimeKit.Text;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Threading.Tasks;

namespace webApp.Services
{
  // source http://6figuredev.com/technology/generic-repository-dependency-injection-with-net-core/

  public interface IRepository<T>
  {
    T Get<TKey>(TKey id);
    IQueryable<T> GetAll();
    void Add(T entity);
    void Update(T entity);
  }

  public class Repository<T> : IRepository<T> where T : class
  {
    protected readonly DbContext Context;
    protected DbSet<T> DbSet;

    public Repository(DbContext context)
    {
      Context = context;
      DbSet = context.Set<T>();
    }

    public void Add(T entity)
    {
      Context.Set<T>().Add(entity);

      Save();
    }

    public T Get<TKey>(TKey id)
    {
      return DbSet.Find(id);
    }

    public IQueryable<T> GetAll()
    {
      return DbSet;
    }

    public void Update(T entity)
    {
      Save();
    }

    private void Save()
    {
      Context.SaveChanges();
    }
  }


  public class EmailService : IEmailService
  {
    private readonly EmailConfig ec;

    public EmailService(IOptions<EmailConfig> emailConfig)
    {
      this.ec = emailConfig.Value;
    }

    public async Task SendEmailAsync(String email, String subject, String message)
    {
      try
      {
        var emailMessage = new MimeMessage();

        emailMessage.From.Clear();
        emailMessage.From.Add(new MailboxAddress(ec.FromName, ec.FromAddress));

        emailMessage.To.Clear();
        emailMessage.To.Add(new MailboxAddress("", email));
        emailMessage.Subject = subject;
        emailMessage.Body = new TextPart(TextFormat.Html) { Text = message };

        using (var client = new SmtpClient())
        {
          client.LocalDomain = ec.LocalDomain;

          await client.ConnectAsync(ec.MailServerAddress, Convert.ToInt32(ec.MailServerPort), SecureSocketOptions.Auto).ConfigureAwait(false);
          await client.AuthenticateAsync(new NetworkCredential(ec.UserId, ec.UserPassword));
          await client.SendAsync(emailMessage).ConfigureAwait(false);
          await client.DisconnectAsync(true).ConfigureAwait(false);
        }
      }
      catch (Exception ex)
      {
        Console.WriteLine(ex.Message);
      }
    }


    //public Task SendSmsAsync(string number, string message)
    //{
    //  // Plug in your SMS service here to send a text message.
    //  return Task.FromResult(0);
    //}
  }
}
