<?php

use \KevBaldwyn\Image\Image;
use \KevBaldwyn\Image\Providers\ProviderInterface;
use \KevBaldwyn\Image\Cache\CacherInterface;
use \Mockery as m;

class ImageTest extends \PHPUnit_Framework_TestCase {

	public function testCallBacksModifyImagePath()
	{
		$image = new Image(static::mockProvider(), 100, '/image-server', static::mockCacher());
		$image->addCallback(Image::CALLBACK_MODIFY_IMG_PATH, function($path) {
			return '/prepend' . $path;
		});
		$image->addCallback(Image::CALLBACK_MODIFY_IMG_PATH, function($path) {
			return $path . '/append';
		});

		$this->assertSame('/prepend/assets/image.jpg/append', $image->getImagePath());
	}


	public function testPathOptions()
	{
		$image = new Image(static::mockProvider(), 100, '/image-server', static::mockCacher());
		// resize
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=resize,400,200', 
			$image->path('/path/to/image.jpg', 'resize', 400, 200)->__toString()
		);
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=resize,400,200,1', 
			$image->path('/path/to/image.jpg', 'resize', 400, 200, 1)->__toString()
		);
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=resize,400', 
			$image->path('/path/to/image.jpg', 'resize', 400)->__toString()
		);

		// resizeCrop
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=resizeCrop,400,200', 
			$image->path('/path/to/image.jpg', 'resizeCrop', 400, 200)->__toString()
		);
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=resizeCrop,400,200,center,middle', 
			$image->path('/path/to/image.jpg', 'resizeCrop', 400, 200, 'center', 'middle')->__toString()
		);

		// crop
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=crop,400,200', 
			$image->path('/path/to/image.jpg', 'crop', 400, 200)->__toString()
		);
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=crop,400,200,left,top', 
			$image->path('/path/to/image.jpg', 'crop', 400, 200, 'left', 'top')->__toString()
		);
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=crop,400,200,20,50%', 
			$image->path('/path/to/image.jpg', 'crop', 400, 200, 20, '50%')->__toString()
		);
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=crop,50%,50%', 
			$image->path('/path/to/image.jpg', 'crop', '50%', '50%')->__toString()
		);
	}


	public function testResponsivePathOptions()
	{
		$image = new Image(static::mockProvider(), 100, '/image-server', static::mockCacher());
		$this->assertSame(
			'/image-server?img=/path/to/image.jpg&transform=resizeCrop,800,600;max-width=400:resize,400&responsive=true', 
			$image->path('/path/to/image.jpg', 'resizeCrop', 800, 600)
				->responsive('max-width=400', 'resize', '400')
				->__toString()
		);
	}


	public function testCreateImage()
	{
		$cacher = static::mockCacher();
		$cacher->shouldReceive('exists')->andReturn(false);
		$cacher->shouldReceive('put');
		$cacher->shouldReceive('getSrcPath')->andReturn(__DIR__ . '/assets/image.jpg');

		$image = new Image(static::mockProvider(), 100, '/image-server', $cacher);
		$data = $image->getImageData();

		$this->assertSame('image/jpeg', $data['mime']);
		$this->assertSame(static::base64imageStr(), base64_encode($data['data']));
	}


	public function testCreateImageFromCache()
	{
		$cacher = static::mockCacher();
		$cacher->shouldReceive('exists')->andReturn(true);
		$cacher->shouldReceive('getImageData')->andReturn(array(
			'data' => static::base64imageStr(), 
			'mime' => 'image/jpeg'
		));

		$image = new Image(static::mockProvider(), 100, '/image-server', $cacher);
		$data = $image->getImageData();

		$this->assertTrue($image->isFromCache());
		$this->assertSame('image/jpeg', $data['mime']);
		$this->assertSame(static::base64imageStr(), $data['data']);
	}
	

	public static function mockProvider()
	{
		$provider = m::mock('\KevBaldwyn\Image\Providers\ProviderInterface');
		$provider->shouldReceive('getVarImage')->andReturn('img');
		$provider->shouldReceive('getVarTransform')->andReturn('transform');
		$provider->shouldReceive('getVarResponsiveFlag')->andReturn('responsive');
		$provider->shouldReceive('getQueryStringData')->with('img')->andReturn('/assets/image.jpg');
		$provider->shouldReceive('getQueryStringData')->with('transform')->andReturn('resizeCrop,800,600');
		$provider->shouldReceive('getQueryStringData')->with('responsive')->andReturn(null);
		$provider->shouldReceive('publicPath')->andReturn(__DIR__);
		$provider->shouldReceive('getWorkerName')->andReturn('Gd');

		return $provider;
	}


	public static function mockCacher()
	{
		$cacher = m::mock('\KevBaldwyn\Image\Cache\CacherInterface');
		$cacher->shouldReceive('init');
		return $cacher;
	}


	public static function base64imageStr()
	{
		return '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';
	}

}