﻿using System;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using RoutedLocalizationExample.Utils;

namespace RoutedLocalizationExample.Tests
{
    [TestClass]
    public class LocalizationUrlHandlerTests
    {
        private IList<string> GetMockedControllersNamesList()
        {
            return new List<string> { "Home", "Person", "Department" };
        }

        private IList<string> GetMockedUserLangs()
        {
            return new List<string> { "en-EN", "ru-RU" };
        }

        [TestMethod]
        public void LocaliseFreeRoute_Success()
        {
            var testedUri = new Uri("http://valid-domain-name.com");
            var controllersList = GetMockedControllersNamesList();
            var langsList = GetMockedUserLangs();

            var res = LocalizationHelper.GetLocalisedUrl(testedUri, controllersList, langsList);
            Assert.AreEqual("http://valid-domain-name.com:80/en/", res);
        }

        [TestMethod]
        public void LocaliseInitialRouteWithBadLocale_Success()
        {
            var testedUri = new Uri("http://valid-domain-name.com/wronglocale");
            var controllersList = GetMockedControllersNamesList();
            var langsList = GetMockedUserLangs();

            var res = LocalizationHelper.GetLocalisedUrl(testedUri, controllersList, langsList);
            Assert.AreEqual("http://valid-domain-name.com:80/en", res);
        }

        [TestMethod]
        public void LocaliseNormalRoute_Success()
        {
            var testedUri = new Uri("http://valid-domain-name.com/Home/Index");
            var controllersList = GetMockedControllersNamesList();
            var langsList = GetMockedUserLangs();

            var res = LocalizationHelper.GetLocalisedUrl(testedUri, controllersList, langsList);
            Assert.AreEqual("http://valid-domain-name.com:80/en/Home/Index", res);
        }

        [TestMethod]
        public void LocaliseBadLocaleRoute_Success()
        {
            var testedUri = new Uri("http://valid-domain-name.com/wrongLocale/Home/Index");
            var controllersList = GetMockedControllersNamesList();
            var langsList = GetMockedUserLangs();

            var res = LocalizationHelper.GetLocalisedUrl(testedUri, controllersList, langsList);
            Assert.AreEqual("http://valid-domain-name.com:80/en/Home/Index", res);
        }

        [TestMethod]
        public void LocaliseRouteWithNoLangs_Success()
        {
            // Обработка роута без передачи заголовка с языками (такой выполняется поисковыми роботами)
            var testedUri = new Uri("http://valid-domain-name.com/Home/Index");
            var controllersList = GetMockedControllersNamesList();
            IList<string> langsList = null;

            var res = LocalizationHelper.GetLocalisedUrl(testedUri, controllersList, langsList);
            Assert.AreEqual("http://valid-domain-name.com:80/en/Home/Index", res);
        }

              [TestMethod]
        public void NoSupportedLocaleInLangs_Success()
        {
            // Обработка роута без передачи заголовка с языками (такой выполняется поисковыми роботами)
            var testedUri = new Uri("http://valid-domain-name.com/");
            var controllersList = GetMockedControllersNamesList();
            IList<string> langsList = new List<string> { "ch-CH" };

            var res = LocalizationHelper.GetLocalisedUrl(testedUri, controllersList, langsList);
            Assert.AreEqual("http://valid-domain-name.com:80/en/", res);
        }
    }
}
