# Querying

This section of the documentation will take you through how to build up queries, be they
Select, Insert, Update or Delete. But first, a little background so you can understand how it
all fits together.

## Contents

- [Fluency](#fluency)
- [Understanding Traits Used](#understanding-traits-used)
- [SELECT Queries](#select-queries)
    - [select()](#select)
    - [from()](#from)
    - [join()](#join)
    - [where()](#where)
    - [orderBy()](#orderBy)
    - [groupBy()](#groupBy)
    - [having()](#having)
    - [limit()](#limit)
    - [offset()](#offset)
- [INSERT Queries](#insert-queries)
    - [table()](#table)
    - [values()](#values)
- [UPDATE Queries](#update-queries)
    - [table()](#table)
    - [values()](#values)
    - [where()](#where)
    - [limit()](#limit)
- [DELETE Queries](#delete-queries)
    - [table()](#table)
    - [values()](#values)
    - [where()](#where)
    - [limit()](#limit)
- [All Tables Referenced](#all-tables-referenced)
- [Non Standard Queries](#non-standard-queries)
- [Query Flags](#query-flags)

## Fluency

S10\SQL aims to provide as fluent and logical interface to building queries as possible.

To that end, it ensures the following:

- All methods can be chained
- All methods can be used as getters, or setters depending on input parameters

## Understanding Traits Used

S10\SQL makes heavy use of Traits to share functionality between different Query types. The Traits
that are defined are:

| Trait | Description |
|-------|-------------|
| Where | Used for Selects, Updates and Deletes, this trait allows you to specify conditions in a query (WHERE ... portions). |
| Having | Used for Selects, this uses the same engine as Where, allowing the same complex conditions for HAVING |
| Pagination | Offers LIMIT and OFFSET defining functions, in an database provider safe way |
| TableName | Used for Insert, Update and Delete to specify the name of the table we're operating on |
| Values | Used for Insert and Update, allows the safe application of input to a query |

Extracting the top of the Select classes source code, you can see these traits at work:

```php
class Select extends Query
{
    use Where;
    use Having;
    use Paginate;
...
}
```

What this means is wherever you see where(), or having(), or values(), it's **the same function being called**. Which means
all the power of SELECT where() conditions are applied to any other query that uses them (like Update for instance).

## SELECT

The query that you're likely to use most often. Here's a basic example:

```php
$query = (new Solution10\SQL\Select())
    ->select('*')
    ->from('users')
    ->limit(25);
```

And here's a complex one:

```php
$query = (new Solution10\SQL\Select)
    ->select('*')
    ->from('users')
    ->join('locations', 'users.location_id', '=', 'locations.id')
    ->where('users.name', '=', 'Alex')
    ->orWhere('users.name', '=', 'Lucie')
    ->where(function (ConditionBuilder $query) {
        $query
            ->andWith('locations.city', '=', 'London')
            ->andWith('locations.country', '=', 'GB');
    })
    ->orWhere(function (ConditionBuilder $query) {
        $query
            ->andWith('locations.city', '=', 'Toronto')
            ->andWith('locations.country', '=', 'CA')
            ->orWith(function (ConditionBuilder $query) {
                $query->andWith('users.active', '!=', true);
            });
    })
    ->orderBy('name', 'ASC')
    ->limit(25)
    ->offset(5)
;
```

And that one isn't even using HAVING or GROUP BY! Hopefully you can see the kind of power at your
disposal though, nested WHERE conditions, pagination and JOINs are all available. The SQL generated by
that monster incidentally looks like this:

```sql
SELECT *
FROM "users"
JOIN "locations" ON "users"."location_id" = "locations"."id"
WHERE
    "users"."name" = ?
    OR "users"."name" = ?
    AND (
        "locations"."city" = ?
        AND "locations"."country" = ?
    )
    OR (
        "locations"."city" = ?
        AND "locations"."country" = ?
        OR (
            "users"."active" != ?
        )
    )
ORDER BY "name" ASC
LIMIT 5, 25
```

You'll notice that the limit() and offset() have been combined into the SQL standard {offset},{limit} format
and all of the table + column names have been appropriately quoted.

### select()

Chooses which columns to pull from the table you're querying.

Setting selects:

```php
$q = new Solution10\SQL\Select();

// Simple single column:
$q->select('name');

// Multiple columns:
$q->select(['name', 'location']);

// Single column with an alias:
$q->select('birthday', 'dob');

// Multiple columns with aliases:
$q->select([
    'dob' => 'birthday',
    'my_aliased_name' => 'name',
]);
```

select() is **additive**. This means it will append columns from subsequent calls onto the same query. If you
wish to wipe the select portion of the query and start again, use `$q->resetSelect()`.

Returning the defined selects as an array:

```php
$selects = $q->select();
```

Fetching only the select portion of the query as a string:

```php
$selectString = $q->buildSelectSQL();
```

Resetting (emptying) the select portion of the query:

```php
$q->resetSelect();
```

### from()

Selects which tables to pull data from.

Setting froms:

```php
$q = new Solution10\SQL\Select();

// Simple table name
$q->from('users');

// Table with an alias:
$q->from('locations', 'loc');
```

from() is **additive**. This means it will append tables from subsequent calls onto the same query. If you
wish to wipe the from portion of the query and start again, use `$q->resetFrom()`.

Returning the defined from tables as an array:

```php
$tables = $q->from();
```

Fetching only the FROM portion of the query as a string:

```php
$fromString = $q->buildFromSQL();
```

Resetting (emptying) the from portion of the query:

```php
$q->resetFrom();
```

### join()

There are three join functions:

- `join()`: an "inner" JOIN
- `leftJoin()`: a left JOIN
- `rightJoin()`: a right JOIN

They all take the same arguments; `join(table-to-join, left-join-column, operator, right-join-column)`

```php
$q->join('locations', 'locations.id', '=', 'users.location_id');
```

generates:

```sql
JOIN "locations" ON "locations"."id" = "users"."location_id"
```

Naturally, `leftJoin()` and `rightJoin()` work in the same way:

```php
$q->leftJoin('locations', 'locations.id', '=', 'users.location_id');
$q->rightJoin('locations', 'locations.id', '=', 'users.location_id');
```

generate:

```php
LEFT JOIN "locations" ON "locations"."id" = "users"."location_id"
RIGHT JOIN "locations" ON "locations"."id" = "users"."location_id"
```

You can return all the defined join()'s like so:

```php
$joins = $q->join();
```

Build up only the JOIN SQL string:

```php
$joinSQL = $q->buildJoinSQL();
```

And reset all the joins:

```php
$q->resetJoins();
```

### where()

Apply conditions onto the query. This can be simple, or nested to any depth.

```php
$q = new Solution10\SQL\Select();
$q->where('name', '=', 'Alex');
```

where() is an AND join between conditions, so:

```php
$q
    ->where('name', '=', 'Alex')
    ->where('age', '>=', 27);
```

becomes:

```sql
WHERE "name" = ? AND "age" > ?
```

You can specify OR conditions with `orWhere()`:

```php
$q
    ->where('name', '=', 'Alex')
    ->orWhere('name', '=', 'Lucie');
```

which becomes:

```sql
WHERE "name" = ? OR "name" = ?
```

You can also 'nest' queries using closures:

```php
$q->where(function(Solution10\SQL\ConditionBuilder $subConditions) {
    $subConditions
        ->andWith('name', '=', 'Alex')
        ->orWith('name', '=', 'Lucie')
});
```

Which will wrap the conditions in a group:

```sql
WHERE (
    "name" = ?
    OR "name" = ?
)
```

Which allows you to do complex stuff like:

```php
$q->where(function (ConditionBuilder $query) {
        $query
            ->andWith('locations.city', '=', 'London')
            ->andWith('locations.country', '=', 'GB');
    })
    ->orWhere(function (ConditionBuilder $query) {
        $query
            ->andWith('locations.city', '=', 'Toronto')
            ->andWith('locations.country', '=', 'CA')
            ->orWith(function (ConditionBuilder $query) {
                $query->andWith('users.active', '!=', true);
            });
    })
```

which would generate:

```sql
WHERE
    (
        "locations"."city" = ?
        AND "locations"."country" = ?
    )
    OR (
        "locations"."city" = ?
        AND "locations"."country" = ?
        OR (
            "users"."active" != ?
        )
    )
```

where() and orWhere() are **additive**; Conditions will be appended to the query.

Returning the defined conditions as an array:

```php
$conditions = $q->where();
```

Fetching only the WHERE portion of the query as a string:

```php
$fromString = $q->buildWhereSQL();
```

Resetting (emptying) the conditions of the query:

```php
$q->resetWhere();
```

### orderBy()

Define the columns and direction to order with.

```php
$q = new Solution10\SQL\Select();

$q
    ->orderBy('name', 'ASC')
    ->orderBy('id', 'DESC');
```

generates:

```sql
ORDER BY "name" ASC, "id" DESC
```

Returning the defined ORDER BY's as an array:

```php
$ordering = $q->orderBy();
```

Fetching only the ORDER BY portion of the query as a string:

```php
$orderByString = $q->buildOrderBySQL();
```

Resetting (emptying) the ORDER portion of the query:

```php
$q->resetOrderBy();
```

### groupBy()

Define the columns to groupBy.

```php
$q = new Solution10\SQL\Select();

$q
    ->groupBy('user.name')
    ->groupBy('user.location');
```

generates:

```sql
GROUP BY "user"."name", "user"."location"
```

Returning the defined GROUP BY's as an array:

```php
$groupings = $q->groupBy();
```

Fetching only the GROUP BY portion of the query as a string:

```php
$groupByString = $q->buildGroupBySQL();
```

Resetting (emptying) the GROUP BY portion of the query:

```php
$q->resetGroupBy();
```

### having()

Having behaves and has exactly the same feature set as where() (including an `orHaving()` function).

You can also nest in exactly the same way, here's an example:

```php
$q
    ->having(function (ConditionBuilder $query) {
        $query
            ->andWith('city', '=', 'London')
            ->andWith('country', '=', 'GB');
    })
    ->orHaving(function (ConditionBuilder $query) {
        $query
            ->andWith('city', '=', 'Toronto')
            ->andWith('country', '=', 'CA')
            ->orWith(function (ConditionBuilder $query) {
                $query->andWith('active', '!=', true);
            });
    });
```

Notice how the nesting uses the same "with" prefixed functions.

Returning the defined HAVING's as an array:

```php
$having = $q->having();
```

Fetching only the HAVING portion of the query as a string:

```php
$havingString = $q->buildHavingSQL();
```

Resetting (emptying) the HAVING portion of the query:

```php
$q->resetHaving();
```

### limit()

Sets the number of results to return.

```php
$q = new Solution10\SQL\Select();

$q->limit(10);
```

generates:

```sql
LIMIT 10
```

Returning the defined LIMIT as an array:

```php
$limit = $q->limit();
```

Fetching only the LIMIT portion of the query as a string:

```php
$pagination = $q->buildPaginateSQL();
```

Resetting (emptying) the LIMIT portion of the query:

```php
$q->resetLimit();
```

### offset()

Sets where to start counting results from.

```php
$q = new Solution10\SQL\Select();

$q->limit(10);
$q->offset(25);
```

generates:

```sql
LIMIT 25,10
```

Returning the defined offset as an array:

```php
$offset = $q->offset();
```

Fetching only the LIMIT portion of the query as a string:

```php
$pagination = $q->buildPaginateSQL();
```

Resetting (emptying) the offset portion of the query:

```php
$q->resetOffset();
```


## INSERT Queries

### table()

Sets which table we're inserting into.

```php
$q = new Solution10\SQL\Insert();

$q->table('users');
```

generates:

```sql
INSERT INTO "users"
```

Returning the table name:

```php
$table = $q->table();
```

Resetting (emptying) the table portion of the query:

```php
$q->resetTable();
```

### values()

Key value pairs of values for the query. If the value has already been set, the second call to values() will
overwrite the old one.

```php
$q->values(['name' => 'Alex', 'location' => 27]);
```

Returning the values:

```php
$values = $q->values();
```

## UPDATE Queries

### table()

Sets which table we're inserting into.

```php
$q = new Solution10\SQL\Insert();

$q->table('users');
```

generates:

```sql
INSERT INTO "users"
```

Returning the table name:

```php
$table = $q->table();
```

Resetting (emptying) the table portion of the query:

```php
$q->resetTable();
```

### values()

Key value pairs of values for the query. If the value has already been set, the second call to values() will
overwrite the old one.

```php
$q->values(['name' => 'Alex', 'location' => 27]);
```

Returning the values:

```php
$values = $q->values();
```

### where()

See [SELECT - where()](#where).

### limit()

See [SELECT - limit()](#limit).

## DELETE Queries

### table()

Sets which table we're inserting into.

```php
$q = new Solution10\SQL\Insert();

$q->table('users');
```

generates:

```sql
INSERT INTO "users"
```

Returning the table name:

```php
$table = $q->table();
```

Resetting (emptying) the table portion of the query:

```php
$q->resetTable();
```

### where()

See [SELECT - where()](#where).

### limit()

See [SELECT - limit()](#limit).

## All Tables Referenced

All of the queries in S10\SQL also give you the ability to ask for a list of tables that the
query is operating on. This combines INTO, FROM, JOIN etc into a single array, so you can know
exactly what the query will be touching.

The method signature is the same for all query types:

```php
$tables = $q->allTablesReferenced();
```

**Note**: this will return the full table name and NOT any aliases you might have set up. If you need the aliases,
you'll have to talk to the `table()`, `from()`, `join()` etc functions directly.

## Non Standard Queries

What about if you want to use something like `REPLACE INTO` which is a non-standard statement, but extremely useful?

Well one option is to subclass the existing queries. `UPDATE` is very similar to `REPLACE INTO` so you could simply
subclass and modify the `sql()` method on it.

However there's an easier way. You can modify the "base statement" of any of the built in queries like so:

```php
$q = new Update();
$q->queryBaseStatement('REPLACE INTO');
```

This also allows you to do things like `SELECT DISTINCT`:

```php
$q = new Select();
$q->queryBaseStatement('SELECT DISTINCT');
```

**Note**: the input to queryBaseStatement is NOT escaped or sanitised in any way. Be extremely careful how you craft the
string which will go in here.

## Query Flags

Sometimes you might have additional metadata that you want to pass along with the query, but that is unrelated to
the generation of the SQL. For example, a cache time-to-live value which can be used by your ORM.

S10\SQL provides a `flag()` function on all queries which allows you to set this extra metadata.

```php
$q = new Select;
$q->flag('ttl', 30);

$flagValue = $q->flag('ttl');
```

You can also set and get multiple flags at once:

```php
$q->flags([
    'ttl' => 30,
    'cacheKey' => 'mykey',
    'model' => new User
]);

$allFlags = $q->flags();
```

And delete flags entirely:

```php
$q->deleteFlag('ttl');
```
