﻿using System;
using System.ComponentModel;
using System.Threading;

namespace HttpCaching.Models
{
    public abstract class ChangeObserver<TItem>
        where TItem : class, INotifyPropertyChanged
    {
        #region Field

        private readonly TItem item;

        private Change lastChange;

        #endregion

        #region Property

        public TItem Item
        {
            get { return this.item; }
        }

        public Change LastChange
        {
            get { return this.lastChange; }
        }

        #endregion

        #region Constructure

        public ChangeObserver(TItem item)
            : this(item, null)
        {
        }

        public ChangeObserver(TItem item, Change lastChange)
        {
            if (item == null) throw new ArgumentNullException("item");

            this.item = item;
            this.item.PropertyChanged += this.item_PropertyChanged;
            this.lastChange = lastChange;
        }

        #endregion

        #region Event Handler

        private void item_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            // Update the latest change information whenever the change occurs.
            Interlocked.Exchange(ref this.lastChange, new Change(DateTime.UtcNow, Guid.NewGuid()));
        }

        #endregion
    }

    public class Change
    {
        private DateTime lastModifiedUtc;
        private Guid eTag;

        public DateTime LastModifiedUtc
        {
            get { return this.lastModifiedUtc; }
        }

        public Guid ETag
        {
            get { return this.eTag; }
        }

        public Change(DateTime lastModifiedUtc, Guid eTag)
        {
            this.lastModifiedUtc = lastModifiedUtc;
            this.eTag = eTag;
        }
    }
}