﻿using System;
using System.Collections.Generic;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Web.Http;

namespace HttpCaching.Controllers
{
    public static class ValuesControllerExtensions
    {
        public static TValue EndIfNull<TValue>(this TValue value) 
            where TValue : class
        {
            if (value == null)
                throw new HttpResponseException(HttpStatusCode.BadRequest);

            return value;
        }

        public static TValue EndIfNotFound<TKey, TValue>(this IDictionary<TKey, TValue> dictionary, TKey key)
        {
            if (dictionary == null)
                throw new HttpResponseException(HttpStatusCode.NotFound);

            TValue value;

            if (dictionary.TryGetValue(key, out value))
                return value;
            else
                throw new HttpResponseException(HttpStatusCode.NotFound);
        }
         
        public static void EndIfNotModified(this HttpRequestMessage request, DateTime lastModified)
        {
            if (request == null)
                throw new ArgumentNullException("request");

            var ifModifiedSince = request.Headers.IfModifiedSince;

            if (ifModifiedSince != null && ifModifiedSince.Value.DateTime >= lastModified)
                throw new HttpResponseException(HttpStatusCode.NotModified);
        }

        public static void EndIfNoneMatch(this HttpRequestMessage request, Guid eTag)
        {
            if (request == null)
                throw new ArgumentNullException("request");

            var ifNoneMatch = request.Headers.IfNoneMatch;

            if (ifNoneMatch == null || ifNoneMatch.Count == 0)
                return;

            var entityTag = new EntityTagHeaderValue(string.Format("\"{0}\"", eTag), false);

            if (ifNoneMatch.Contains(entityTag))
                throw new HttpResponseException(HttpStatusCode.NotModified);
        }

        public static HttpResponseMessage CreateResponse<T>(this HttpRequestMessage request,
            HttpStatusCode statusCode, T value, DateTime lastModified, TimeSpan expires)
        {
            if (request == null)
                throw new ArgumentNullException("request");

            var response = request.CreateResponse<T>(statusCode, value);

            response.Headers.CacheControl = new CacheControlHeaderValue();
            response.Content.Headers.LastModified = new DateTimeOffset(lastModified);
            response.Content.Headers.Expires = new DateTimeOffset(DateTime.UtcNow + expires.Duration());

            return response;
        }

        public static HttpResponseMessage CreateResponse<T>(this HttpRequestMessage request,
            HttpStatusCode statusCode, T value, Guid eTag, TimeSpan expires)
        {
            if (request == null)
                throw new ArgumentNullException("request");

            var response = request.CreateResponse<T>(statusCode, value);

            response.Headers.CacheControl = new CacheControlHeaderValue();
            response.Headers.ETag = new EntityTagHeaderValue(string.Format("\"{0}\"", eTag), false);
            response.Content.Headers.Expires = new DateTimeOffset(DateTime.UtcNow + expires.Duration());

            return response;
        }
    }
}