﻿using HttpCaching.Models;
using System;
using System.Collections.Concurrent;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Web;
using System.Web.Helpers;
using System.Web.Http;

namespace HttpCaching.Controllers
{
    public class ValuesController : ApiController
    {
        #region Fields

        public static readonly ConcurrentDictionary<Guid, EmployeeChangeObserver> Employees;
        public static readonly TimeSpan DefaultExpires;

        #endregion

        #region Constructor

        static ValuesController()
        {
            // Read JSON data from text file as the default content of the dictionary.
            var fileInfo = new FileInfo(HttpContext.Current.Server.MapPath("~/App_Data/Valve.txt"));
            var lastChange = new Change(fileInfo.LastWriteTimeUtc, Guid.NewGuid());
            var employees = Json.Decode<Employee[]>(File.ReadAllText(fileInfo.FullName)).
                ToDictionary(c => c.ID, c => new EmployeeChangeObserver(c, lastChange));

            Employees = new ConcurrentDictionary<Guid, EmployeeChangeObserver>(employees);

            // The default expiration time in clients' cache is 1 minute.
            DefaultExpires = TimeSpan.FromMinutes(1);
        }

        #endregion

        #region Methods

        [HttpGet]
        public HttpResponseMessage Select(Guid id)
        {
            var employee = Employees.EndIfNotFound(id);
            var lastChange = employee.LastChange;

            // If change information is not available, will end here.
            if (lastChange == null)
                return base.Request.CreateResponse(HttpStatusCode.OK, employee.Item);

            // If nothing changed, will raise an HttpResponseException in Status 304.
            base.Request.EndIfNotModified(lastChange.LastModifiedUtc);

            // Give the latest change information.
            return base.Request.CreateResponse(HttpStatusCode.OK, employee.Item,
                lastChange.LastModifiedUtc, DefaultExpires);
        }

        [HttpPost]
        public void Create([FromBody]Employee employee)
        {
            employee.EndIfNull();
            
            if (!Employees.TryAdd(employee.ID, new EmployeeChangeObserver(employee)))
                throw new HttpResponseException(HttpStatusCode.Conflict);
        }

        [HttpPut]
        public void Update([FromBody]Employee employee)
        {
            employee.EndIfNull();

            var employeeOld = Employees.EndIfNotFound(employee.ID).Item;

            employeeOld.Alias = employee.Alias;
            employeeOld.Description = employee.Description;
            employeeOld.FirstName = employee.FirstName;
            employeeOld.LastName = employee.LastName;
            employeeOld.Sex = employee.Sex;
            employeeOld.SteamID = employee.SteamID;
        }

        #endregion
    }
}
