<?php 

/**
* Simple PHP image thumbnail generator class.
*
* @author Kabelo Nkakala
*/

class Thumbnail {
	
	/**
	* Image file
	* @var $file 
	*/
	protected $file;
	
	/**
	* File name 
	* @var $name 
	*/
	protected $name;
	
	/**
	* Mime type 
	* @var $mimeType 
	*/
	protected $mimeType;
	
	/**
	* File destination 
	* @var $destination
	*/
	protected $destImage;
	
	/**
	* Thumbnail width 
	* @var $width
	*/
	protected $width;
	
	/**
	* Thumbnail height 
	* @var $height 
	*/ 
	protected $height;
	
	/**
	* Temporary Image (thumbnail) 
	* @var $tmpImage 
	*/
	protected $tmpImage;
	
	/** 
	* Supported mime types 
	* @array $supported 
	*/ 
	protected $supported = array('image/gif', 'image/jpeg', 'image/png');
	
	/**
	* Create the image thumbnail 
	* 
	* @param string $file
	* @param string $name 
	* @param int/float $maxWidth
	* @param string $destination 
	*/ 
	function __construct($file, $name, $maxWidth, $destination) {
		/* 
			Construct the thumbnail 
		*/
		$this->file = $file;
		
		$this->name = $name;
		
		$this->width = $maxWidth;
		
		try {
		
			$this->getMimeType($this->file);
			
		} catch (Exception $e) {
			
			var_dump($e->getMessage);
		}
		
		if (in_array($this->mimeType, $this->supported, true)) {
			
			try {
				
				switch ($this->mimeType) {
				
					case 'image/gif': 
						$img = imagecreatefromgif($this->file); 
					break;
					
					case 'image/jpeg': 
						$img = imagecreatefromjpeg($this->file);
					break;
					
					case 'image/png': 
						$img = imagecreatefrompng($this->file);
					break;
				
				}
				
				$this->oWidth = imagesx($img);
				$this->oHeight = imagesy($img);
				
				$this->calcHeight($this->width, $this->oWidth, $this->oHeight);
				$this->tmpImage = imagecreatetruecolor($this->width, $this->height); 
				
				if (!is_dir($destination)) {
					
					mkdir($destination, 0777);
				
				}
				
				$this->destImage = $destination . $this->name;
			
				try {
				
					imagecopyresampled($this->tmpImage, $img, 0, 0, 0, 0, $this->width, $this->height, $this->oWidth, $this->oHeight);
					imagejpeg($this->tmpImage, $this->destImage, 100); 
					
				} catch (Exception $e) {
					
					var_dump($e->getMessage);
					
				}
			
			} catch (Exception $e) {
				
				var_dump($e->getMessage);
				
			}
			
			imagedestroy($this->tmpImage);
			
		}	
		
	}
	
	/**
	* Get the image file's mime type 
	*
	* @return TRUE if mime type could be determined 
	*/
	protected function getMimeType($file) {
		/*
			Get the file's mimeType 
		*/
		$finfo = new finfo(FILEINFO_MIME);
		$mime = explode(';', $finfo->file($file));
		$this->mimeType = $mime[0];
		
	}
	
	/**
	* Calculate the thumbnail's height 
	* 
	* @param int/float $this->width
	* @return TRUE on success 
	*/ 
	protected function calcHeight($thumbWidth, $oWidth, $oHeight) {
		/* 
			Calculate the thumbnail's height 
		*/
		$this->height = floor($oHeight * ($thumbWidth / $oWidth));
		
	}
	
}	
	
	