﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Text;
using System.Web;
using System.Web.UI;

namespace Demo.Classes
{
    public class CrashReport
    {
        #region Properties

        public String Body { get; set; }
        public Exception Exception { get; set; }
        public String Title { get; set; }

        #endregion

        #region Methods (construction)

        public CrashReport(Exception ex, String comment = null)
        {
            Exception = ex;
            HttpContext context = HttpContext.Current;

            StringBuilder sb = new StringBuilder();

            sb.Append("<div id='crash-report'>");
            sb.AppendFormat("<h2>{0}: {1}</h2>", ex.GetType().Name, ex.Message);

            if (comment != null)
                sb.AppendFormat("<p>{0}</p>", comment);

            sb.Append("<table class='style-one'>");
            sb.AppendFormat("<tr><td class='label'>Time</td><td>{0:dddd - MMM d, yyyy h:mm tt} {1} ({2:h:mm tt} UTC)</td></tr>", DateTime.Now, TimeZone.CurrentTimeZone.StandardName, DateTime.UtcNow);
            sb.AppendFormat("<tr><td class='label'>Host</td><td>{0}</td></tr>", context.Request.Url.Host);
            sb.AppendFormat("<tr><td class='label'>Request</td><td>{0}</td></tr>", context.Request.Url);

            if (context.Request.UrlReferrer != null)
                sb.AppendFormat("<tr><td class='label'>Referrer</td><td>{0}</td></tr>", context.Request.UrlReferrer);

            sb.AppendFormat("<tr><td class='label'>User Identity</td><td>{0}</td></tr>", (context.Request.IsAuthenticated ? context.User.Identity.Name : "Anonymous"));
            sb.AppendFormat("<tr><td class='label'>Authentication</td><td>{0}</td></tr>", (context.Request.IsAuthenticated ? context.User.Identity.AuthenticationType : "None"));
            sb.AppendFormat("<tr><td class='label'>IP Address</td><td>{0}</td></tr>", context.Request.UserHostAddress);
            sb.AppendFormat("<tr><td class='label'>Browser</td><td>{0} {1} {2}</td></tr>", context.Request.Browser.Platform, context.Request.Browser.Browser, context.Request.Browser.Version);

            sb.Append(FormatException(ex, this, 1));

            if (context.Request != null)
            {
                sb.Append("<tr><th colspan='2'><h3>Request Variables</h3></th></tr>");

                DescribeValues("Query String", context.Request.QueryString, sb, null, null);
                DescribeValues("Form Variables", context.Request.Form, sb, new[] { "__VIEWSTATE" }, null);
                DescribeValues("Cookies", context.Request.Cookies, sb, null, new[] { "Total_UserPassword" });
                DescribeValues("Server Variables", context.Request.ServerVariables, sb, new[] { "HTTP_COOKIE", "ALL_HTTP", "ALL_RAW" }, new[] { "AUTH_PASSWORD" });
            }

            sb.Append("</table>");
            sb.Append("</div>");

            Body = sb.ToString();
        }

        #endregion

        #region Methods (helpers)

        private static String FormatException(Exception ex, CrashReport report, Int32 depth)
        {
            report.Title = ex.Message;

            var sb = new StringBuilder();

            if (ex.InnerException != null)
                sb.Append(FormatException(ex.InnerException, report, depth - 1));

            sb.AppendFormat("<tr><th colspan='2'><h3>Exception #{0}</h3></th></tr>", Math.Abs(depth) + 1);

            if (!String.IsNullOrEmpty(ex.Source))
                sb.AppendFormat("<tr><td class='label'>Source</td><td>{0}</td></tr>", ex.Source);

            sb.AppendFormat("<tr><td class='label'>Message</td><td>{0}</td></tr>", FormatExceptionMessage(ex));
            sb.AppendFormat("<tr><td class='label'>Stack Trace</td><td>{0}</td></tr>", FormatExceptionStackTrace(ex));

            return sb.ToString();
        }

        private static String FormatExceptionMessage(Exception ex)
        {
            var httpException = ex as HttpException;
            if (httpException != null)
                return String.Format("HTTP Error {0}: {1}", httpException.ErrorCode, httpException.Message);

            var viewStateException = ex as ViewStateException;
            if (viewStateException != null)
                return "View State Error: The server was still processing your last action (slow down). This error sometimes occurs when you perform an action in the system and then perform another action before your first action has finished being processed by the server. (Refer to http://support.microsoft.com/kb/555353)";

            return String.Format("{0}: {1}", ex.GetType().Name, ex.Message);
        }

        private static String FormatExceptionStackTrace(Exception ex)
        {
            String html;

            if (ex.StackTrace != null)
            {
                html = ex.StackTrace.Replace(" at ", "\n<br /> at ").Trim();

                if (html.StartsWith("<br /> at "))
                    html = html.Remove(0, 6);

                html = html.Replace(" in ", "<br>&nbsp;&nbsp;&nbsp;&nbsp;in ");
            }
            else
            {
                html = "&nbsp;";
            }

            return html;
        }

        private static void DescribeValues(String name, NameValueCollection collection, StringBuilder sb, IEnumerable<String> exclude, String[] mask)
        {

            sb.AppendFormat("<tr><td class='label' valign='top' style='white-space:nowrap'>{0}</td>", name);

            if (collection != null && collection.Count > 0)
            {
                List<String> excludeKeys = exclude == null ? new List<String>() : new List<String>(exclude);
                List<String> maskedKeys = mask == null ? new List<String>() : new List<String>(mask);

                sb.Append("<td><table class='namevaluecollection'><tr><th>Key</th><th>Value</th></tr>");

                foreach (String key in collection.Keys)
                {
                    if (excludeKeys.Contains(key))
                        continue;

                    String value = maskedKeys.Contains(key) ? (String.IsNullOrEmpty(collection[key]) ? String.Empty : "***********") : collection[key];

                    sb.AppendFormat("<tr><td>{0}</td><td>{1}</td></tr>", HttpContext.Current.Server.HtmlEncode(key), HttpContext.Current.Server.HtmlEncode(value));
                }

                sb.Append("</table></td></tr>");
            }
            else
            {
                sb.Append("<td>(empty)</td></tr>");
            }
        }

        private static void DescribeValues(String name, HttpCookieCollection collection, StringBuilder sb, IEnumerable<String> exclude, IEnumerable<String> mask)
        {
            sb.AppendFormat("<tr><td class='label' valign='top' style='white-space:nowrap'>{0}</td>", name);

            if (collection != null && collection.Count > 0)
            {
                List<String> excludeKeys = exclude == null ? new List<String>() : new List<String>(exclude);
                List<String> maskedKeys = mask == null ? new List<String>() : new List<String>(mask);

                sb.Append("<td><table class='namevaluecollection'><tr><th>Key</th><th>Value</th></tr>");

                foreach (String key in collection.Keys)
                {
                    if (excludeKeys.Contains(key))
                        continue;

                    if (collection[key] == null)
                        continue;

                    String value = maskedKeys.Contains(key) ? (String.IsNullOrEmpty(collection[key].Value) ? String.Empty : "***********") : collection[key].Value;

                    sb.AppendFormat("<tr><td>{0}</td><td>{1}</td></tr>", HttpContext.Current.Server.HtmlEncode(key), HttpContext.Current.Server.HtmlEncode(value));
                }

                sb.Append("</table></td></tr>");
            }
            else
            {
                sb.Append("<td>(empty)</td></tr>");
            }
        }

        #endregion
    }
}