﻿using System;
using System.Diagnostics;
using System.IO;
using System.Net.Mail;
using System.Text.RegularExpressions;
using System.Web;
using System.Web.Hosting;
using System.Web.Security;
using System.Web.SessionState;
using System.Xml.Linq;

namespace Demo.Classes
{
    public class ApplicationErrorModule : IHttpModule
    {
        private const String DefaultProgrammerEmailAddress = "daniel.miller@insitesystems.com";
        private const String DefaultSenderEmailAddressPrefix = "crash.reports@";

        #region Classes

        public class Settings
        {
            public static class Names
            {
                public static String Enabled = "Demo.ApplicationErrorModule.Enabled";
                
                public static String CrashReportEmailAddress = "Demo.ApplicationErrorModule.CrashReportEmail.Address";
                public static String CrashReportEmailEnabled = "Demo.ApplicationErrorModule.CrashReportEmail.Enabled";
                public static String CrashReportPath = "Demo.ApplicationErrorModule.CrashReportPath";
                public static String CrashReportUrl = "Demo.ApplicationErrorModule.CrashReportUrl";

                /// <summary>
                /// The key used to index the cache includes the session ID (whenever possible) to make it session-safe.
                /// </summary>
                public static String CrashReportKey
                {
                    get
                    {
                        String key = "Demo.ApplicationErrorModule.CrashReport";
                        if (HttpContext.Current.Session != null)
                            key += "." + HttpContext.Current.Session.SessionID;
                        return key;
                    }
                } 
            }

            public static String CrashReportPath
            {
                get { return AppSettingsHelper.GetValue(Names.CrashReportPath, false); }
            }

            public static String CrashReportEmailAddress
            {
                get { return AppSettingsHelper.GetValue(Names.CrashReportEmailAddress, false, DefaultProgrammerEmailAddress); }
            }

            public static Boolean CrashReportEmailEnabled
            {
                get { return AppSettingsHelper.GetValueAsBoolean(Names.CrashReportEmailEnabled, false); }
            }

            public static Boolean CrashReportEnabled
            {
                get { return AppSettingsHelper.GetValueAsBoolean(Names.CrashReportEmailEnabled, false); }
            }

            public static String CrashReportUrl
            {
                get { return AppSettingsHelper.GetValue(Names.CrashReportUrl, false); }
            }

            public static Boolean Enabled
            {
                get { return AppSettingsHelper.GetValueAsBoolean(Names.Enabled, false); }
            }
        }

        #endregion

        #region Fields

        private const String ErrorPagePattern = @"/Error(404|500)?\.aspx$";
        private const String ModuleName = "ApplicationErrorModule";

        #endregion

        #region Properties

        /// <summary>
        /// Returns True if the existingResponse attribute value under system.webServer/httpErrors is set to "Replace".
        /// </summary>
        private static Boolean ReplaceResponse
        {
            get
            {
                XDocument config = XDocument.Load(HostingEnvironment.ApplicationPhysicalPath + "Web.config");
                if (config.Root != null)
                {
                    XElement webServer = config.Root.Element("system.webServer");
                    if (webServer != null)
                    {
                        XElement httpErrors = webServer.Element("httpErrors");
                        if (httpErrors != null)
                        {
                            String existingResponse = httpErrors.Attribute("existingResponse").Value;
                            return existingResponse == "Replace";
                        }
                    }
                }

                return false;
            }
        }

        #endregion

        #region Events

        public delegate void UnhandledExceptionOccurredDelegate(Exception ex);

        /// <summary>
        /// Allow client code to hook the event handler for an unhandled exception. A specific application might need 
        /// to perform a specific function when an error occurs.
        /// </summary>
        public event UnhandledExceptionOccurredDelegate UnhandledExceptionOccurred;

        #endregion

        #region Methods (construction)

        public void Init(HttpApplication application)
        {
            application.Error += Application_Error;
            application.BeginRequest += Application_BeginRequest;
        }

        public void Dispose()
        {
        }

        #endregion

        #region Methods (event handling)

        private void Application_BeginRequest(Object sender, EventArgs e)
        {
            HttpContext.Current.Items[ModuleName] = this;
        }

        private void Application_Error(Object sender, EventArgs e)
        {
            if (!Settings.Enabled)
                return;

            Exception ex = HttpContext.Current.Server.GetLastError();

            if (ex.GetType().Name == "AspNetSessionExpiredException")
            {
                HttpContext.Current.Server.ClearError();
                HttpContext.Current.Response.Redirect(FormsAuthentication.LoginUrl);
            }
            else
            {
                if (UnhandledExceptionOccurred != null)
                    UnhandledExceptionOccurred(ex);

                ExceptionOccurred(ex);
            }
        }

        #endregion

        #region Helpers

        private static void ExceptionOccurred(Exception ex)
        {
            // If an unhandled exception is thrown here then this Application_Error event handler will re-catch it, 
            // wiping out the original exception, and it will not re-throw the exception -- so this line of code does
            // not create an infinite loop (although it might appear to do so).

            // throw new DemoException("What happens to an unhandled exception in ApplicationErrorModule.ExceptionOccurred?");

            // If the current request is itself an error page then we need to allow the exception to pass through.

            HttpRequest request = HttpContext.Current.Request;
            if (Regex.IsMatch(request.Url.AbsolutePath, ErrorPagePattern))
                return;

            // Otherwise, we should handle the exception here

            HttpResponse response = HttpContext.Current.Response;
            CrashReport report = new CrashReport(ex);

            // Save the crash report in the current cache so it is accessible to my custom error pages. The key used to
            // index the cache includes the session ID, so it is session-safe. A session variable set here does not seem
            // to be available in the error page's session state - although the session ID is available there.

            if (HttpContext.Current.Cache != null)
                HttpContext.Current.Cache[Settings.Names.CrashReportKey] = report;

            // Save the crash report on the file system
            
            String path = SaveCrashReport(report, request, null);

            // Send the crash report to the programmers

            try
            {
                SendEmail(report, path);
            }
            catch (Exception sendEmailException)
            {
                SaveCrashReport(new CrashReport(sendEmailException), request, "SendEmail");
            }

            // Write the crash report to the browser if there is no replacement defined for the HTTP response
            
            if (!ReplaceResponse)
            {
                HttpContext.Current.Server.ClearError();

                if (!String.IsNullOrEmpty(Settings.CrashReportUrl))
                {
                    HttpContext.Current.Server.Transfer(Settings.CrashReportUrl);
                }
                else
                {
                    try
                    {
                        response.Clear();
                        response.StatusCode = 500;
                        response.StatusDescription = "Server Error";
                        response.TrySkipIisCustomErrors = true;
                        response.Write(report.Body);
                        response.End();
                    }
                    catch { }
                }
            }
        }

        #endregion

        #region Methods (saving)
		 
        static private String CreateFilePath(String host, String action)
        {
            if (String.IsNullOrEmpty(Settings.CrashReportPath))
                return null;

            const String pattern = @"{0}\{1}\{2:yyyy.MM.dd.HH.mm.ss}{3}.html";
            String path = String.Format(pattern, Settings.CrashReportPath, host, DateTime.UtcNow, String.IsNullOrEmpty(action) ? String.Empty : "." + action);
            return path;
        }

        static private String SaveCrashReport(CrashReport report, HttpRequest request, String action)
        {
            String path = CreateFilePath(request.Url.Host, action);

            if (String.IsNullOrEmpty(path))
                return null;
                
            String directory = Path.GetDirectoryName(path);
            if (directory != null && !Directory.Exists(directory))
                Directory.CreateDirectory(directory);
                
            File.WriteAllText(path, report.Body);

            return path;
        }

	    #endregion

        #region Methods (email notification)
		 
        /// <summary>
        /// Parses the domain from the requested URL, excluding the sub-domain, and uses this to create an email 
        /// address that follows a standard convention. For example: crash.reports@insitesystems.com
        /// </summary>
        private static String CreateSenderEmailAddress()
        {
            // Remove the sub-domain from the host name.
            String host = HttpContext.Current.Request.Url.Host;
            Int32 index = host.LastIndexOf('.', host.LastIndexOf('.') - 1);
            String domain = index < 0 ? host : host.Substring(index + 1);

            return DefaultSenderEmailAddressPrefix + domain;
        }

        public static void SendEmail(CrashReport report, String attachment)
        {
            if (!Settings.CrashReportEmailEnabled)
                return;

            if (String.IsNullOrEmpty(Settings.CrashReportEmailAddress))
                return;

            var email = new MailMessage { Sender = new MailAddress(CreateSenderEmailAddress()) };

            email.To.Add(Settings.CrashReportEmailAddress);
            email.Subject = report.Title;
            email.Body = report.Body;
            email.IsBodyHtml = true;

            if (attachment != null)
                email.Attachments.Add(new Attachment(attachment));

            var smtp = new SmtpClient();
            smtp.Send(email);
        }

        #endregion
    }
}