﻿using DataLayer;
using Microsoft.AspNet.SignalR;
using Microsoft.AspNet.SignalR.Hubs;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Web;

namespace WebParallelProgramming
{
    public class StockBroadcaster
    {
        //Declare and initialze a TimeSpan that will be used by the timer which will be scheduled to send updates to clients
        private readonly TimeSpan _refreshRate = TimeSpan.FromMilliseconds(2000);

        //Declare a random that will help us in deciding which stocks to update, so that we don't have to update all the stocks at the same time.
        private readonly Random _willUpdate = new Random();

        //declare the timer
        private readonly Timer _timer;

        //Used to check if other threads are also updating the same instance, the volatile keyword is used to ensure thread safety.
        private volatile bool _updatingStockPrices = false;

        //This is the lock that will be used to ensure thread safety when updating the stocks.
        private readonly object _updateStockPricesLock = new object();

        //Encapsulates all information about a SignalR connection for a Hub
        private IHubConnectionContext<dynamic> Clients
        {
            get;
            set;
        }

        //Initialize the static Singleton instance. Lazy initialization is used to ensure that the instance creation is threadsafe.
        private readonly static Lazy<StockBroadcaster> _instance = new Lazy<StockBroadcaster>(() => new StockBroadcaster(GlobalHost.ConnectionManager.GetHubContext<StockBroadcasterHub>().Clients));

        //Initialize the dictionary that will be be used to hold the stocks that will be returned to the client. We used
        //the ConcurrentDictionary for thread safety.If you used a Dictionary object make sure to explicitly lock the dictionary before making changes to it.
        private readonly ConcurrentDictionary<string, Stock> _stocks = new ConcurrentDictionary<string, Stock>();

        //Since the constructer is marked as private, then the static _instance is the only instance that can be created from this class
        private StockBroadcaster(IHubConnectionContext<dynamic> clients)
        {
            Clients = clients;

            //Here Fill the Stocks Dictionary that will be broadcasted 
            _stocks.Clear();//Clear the dictionary
            var stocks = new List<Stock>();
            stocks = Stock.GetStocks();
            stocks.AddRange(Stock.GetMoneyStocks());
            stocks.AddRange(Stock.GetPreciousMetals());
            stocks.ForEach(stock => _stocks.TryAdd(stock.Symbol, stock));

            _timer = new Timer(UpdateStocks, null, _refreshRate, _refreshRate); //initialize the timer

        }

        /// <summary>
        /// The singelton instance exposed as a public porperty
        /// </summary>
        public static StockBroadcaster Instance
        {
            get
            {
                return _instance.Value;
            }
        }

        /// <summary>
        /// Will return all the stocks
        /// </summary>
        /// <returns></returns>
        public IEnumerable<Stock> GetAllStocks()
        {
            return _stocks.Values; // here we return the dictionary.
        }

        /// <summary>
        /// Will update all the stocks in the dictionary
        /// </summary>
        /// <param name="state"></param>
        private void UpdateStocks(object state)
        {
            lock (_updateStockPricesLock)
            {
                if (!_updatingStockPrices)
                {
                    List<Stock> stocks = new List<Stock>();

                    _updatingStockPrices = true;

                    foreach (var stock in _stocks.Values)
                    {
                        if (TryUpdateStockPrice(stock))
                        {
                            stocks.Add(stock);

                        }
                    }

                    BroadcastAllStocksPrices(stocks);//Broadcast the updated stocks to the clients

                    _updatingStockPrices = false;
                }
            }
        }

        /// <summary>
        /// Will update an individual stock
        /// </summary>
        /// <param name="stock">The stock to be updated</param>
        /// <returns></returns>
        private bool TryUpdateStockPrice(Stock stock)
        {
            // Randomly choose whether to update this stock or not
            var randomUpdate = _willUpdate.NextDouble();
            if (randomUpdate > 0.3)//To increase the possibility of updating this stock, replace 0.3 with a bigger number, but less than 1
            {
                return false;
            }

            // Update the stock price
            var random = new Random((int)Math.Floor(stock.LastValue));
            double percentChange = random.NextDouble() * 0.1;
            bool isChangePostivie = random.NextDouble() > .51;//To check if we will subtract or add the change value, the random.NextDouble will return a value between 0.0 and 1.0. Thus it is almost a fifty/fifty chance of adding or subtracting
            double changeValue = Math.Round(stock.LastValue * percentChange, 2);
            changeValue = isChangePostivie ? changeValue : -changeValue;

            double newValue = stock.LastValue + changeValue;
            stock.Change = newValue - stock.LastValue;
            stock.LastValue = newValue;
            return true;
        }

        /// <summary>
        /// Will broadcast a single stock to the clients
        /// </summary>
        /// <param name="stock">The stock to broadcast</param>
        private void BroadcastStockPrice(Stock stock)
        {
            Clients.All.updateStockPrice(stock);//This updateStockPrice method will be called at the client side, in the clients' Browsers, with the stock parameter inside it.
        }

        /// <summary>
        /// Will broadcast all updated stocks to the clients
        /// </summary>
        /// <param name="stocks">The stocks to broadcast</param>
        private void BroadcastAllStocksPrices(List<Stock> stocks)
        {
            Clients.All.updateAllStocksPrices(stocks);//This updateStockPrice method will be called at the client side, in the clients' Browsers, with the stocks parameter inside it.
        }

    }
}