/*
 The observer pattern is widely used in client-side JavaScript programming. All the
 browser events (mouseover, keypress, and so on) are examples of the pattern. Another
 name for it is also custom events, meaning events that you create programmatically, as
 opposed to the ones that the browser fires. Yet another name is subscriber/publisher
 pattern.
 The main motivation behind this pattern is to promote loose coupling. Instead of one
 object calling another object’s method, an object subscribes to another object’s specific
 activity and gets notified. The subscriber is also called observer, while the object being
 observed is called publisher or subject. The publisher notifies (calls) all the subscribers
 when an important event occurs and may often pass a message in the form of an event
 object.
 */

/*
 Let’s say
 you have a publisher paper, which publishes a daily newspaper and a monthly magazine.
 A subscriber joe will be notified whenever that happens.
 */

var publisher = {
	subscribers: {
		any: [] // event type: subscribers
	},
	subscribe: function (fn, type) {
		type = type || 'any';
		if (typeof this.subscribers[type] === "undefined") {
			this.subscribers[type] = [];
		}
		this.subscribers[type].push(fn);
	},
	unsubscribe: function (fn, type) {
		this.visitSubscribers('unsubscribe', fn, type);
	},
	publish: function (publication, type) {
		this.visitSubscribers('publish', publication, type);
	},
	visitSubscribers: function (action, arg, type) {
		var pubtype = type || 'any',
			subscribers = this.subscribers[pubtype],
			i,
			max = subscribers.length;
		for (i = 0; i < max; i += 1) {
			if (action === 'publish') {
				subscribers[i](arg);
			} else {
				if (subscribers[i] === arg) {
					subscribers.splice(i, 1);
				}
			}
		}
	}
};

function makePublisher(o) {
	var i;
	for (i in publisher) {
		if (publisher.hasOwnProperty(i) && typeof publisher[i] === "function") {
			o[i] = publisher[i];
		}
	}
	o.subscribers = {any: []};
}

var paper = {
	daily: function () {
		this.publish("big news today");
	},
	monthly: function () {
		this.publish("interesting analysis", "monthly");
	}
};

makePublisher(paper);

var joe = {
	drinkCoffee: function (paper) {
		console.log('Just read ' + paper);
	},
	sundayPreNap: function (monthly) {
		console.log('About to fall asleep reading this ' + monthly);
	}
};

paper.subscribe(joe.drinkCoffee);
paper.subscribe(joe.sundayPreNap, 'monthly');

paper.daily();
paper.daily();
paper.daily();
paper.monthly();

makePublisher(joe);
joe.tweet = function (msg) {
	this.publish(msg);
};

paper.readTweets = function (tweet) {
	alert('Call big meeting! Someone ' + tweet);
};
joe.subscribe(paper.readTweets);

joe.tweet("hated the paper today");

