/*
In this pattern the independent objects (colleagues)
do not communicate directly, but through a mediator object. When one of the
colleagues changes state, it notifies the mediator, and the mediator communicates the
change to any other colleagues that should know about it.
*/

/*
Let’s explore an example of using the mediator pattern. The application will be a game
where two players are given half a minute to compete for who will press a button more
times than the other. Player 1 competes by pressing 1 and player 2 presses 0 (so they
are more comfortable and don’t have to fight over the keyboard). A scoreboard is updated
with the current score.
*/

function Player(name) {
	this.points = 0;
	this.name = name;
}
Player.prototype.play = function () {
	this.points += 1;
	mediator.played();
};

var scoreboard = {
	// HTML element to be updated
	// element: document.getElementById('results'),
	// update the score display
	update: function (score) {
		var i, msg = '';
		for (i in score) {
			if (score.hasOwnProperty(i)) {
				msg += '<p><strong>' + i + '<\/strong>: ';
				msg += score[i];
				msg += '<\/p>';
			}
		}
		console.log(msg);
		//this.element.innerHTML = msg;
	}
};

var mediator = {
	// all the players
	players: {},
	// initialization
	setup: function () {
		var players = this.players;
		players.home = new Player('Home');
		players.guest = new Player('Guest');
	},
	// someone plays, update the score
	played: function () {
		var players = this.players,
			score = {
				Home: players.home.points,
				Guest: players.guest.points
			};
		scoreboard.update(score);
	},
	// handle user interactions
	keypress: function (e) {
		e = e || window.event; // IE
		if (e.which === 49) { // key "1"
			mediator.players.home.play();
			return;
		}
		if (e.which === 48) { // key "0"
			mediator.players.guest.play();
			return;
		}
	}
};

// go!
mediator.setup();

// attach to onkeypress event
window.onkeypress = mediator.keypress;

// game over in 30 seconds
setTimeout(function () {
	window.onkeypress = null;
	alert('Game over!');
}, 30000);