function classify(base, properties){
	/// <summary>Creates a type (i.e. class) that supports prototype-chaining (i.e. inheritance).</summary>
	/// <param name="base" type="Function" optional="true">The base class to extend.</param>
	/// <param name="properties" type="Object" optional="true">The properties of the class, including its constructor and members.</param>
	/// <returns type="Function">The class.</returns>

	// quick-and-dirty method overloading
	properties = (typeof(base) === "object") ? base : properties || {};
	base = (typeof(base) === "function") ? base : Object;

	var basePrototype = base.prototype;
	var derivedPrototype;

	if (Object.create){
		// allow newer browsers to leverage ECMAScript 5 features
		var propertyNames = Object.getOwnPropertyNames(properties);
		var propertyDescriptors = {};

		for (var i = 0, p; p = propertyNames[i]; i++)
			propertyDescriptors[p] = Object.getOwnPropertyDescriptor(properties, p);

		derivedPrototype = Object.create(basePrototype, propertyDescriptors);
	}
	else{
		// provide "shim" for older browsers
		var baseType = function() {};
		baseType.prototype = basePrototype;
		derivedPrototype = new baseType;

		// add enumerable properties
		for (var p in properties)
			if (properties.hasOwnProperty(p))
				derivedPrototype[p] = properties[p];

		// add non-enumerable properties (see https://developer.mozilla.org/en/ECMAScript_DontEnum_attribute)
		if (!{ constructor: true }.propertyIsEnumerable("constructor"))
			for (var i = 0, a = [ "constructor", "hasOwnProperty", "isPrototypeOf", "propertyIsEnumerable", "toLocaleString", "toString", "valueOf" ], p; p = a[i]; i++)
				if (properties.hasOwnProperty(p))
					derivedPrototype[p] = properties[p];
	}

	// build the class
	var derived = properties.hasOwnProperty("constructor") ? properties.constructor : function() { base.apply(this, arguments); };
	derived.prototype = derivedPrototype;
	derived.prototype.constructor = derived;
	derived.prototype.base = derived.base = basePrototype;

	return derived;
}

/* Example 1: Define a minimal class */
var Minimal = classify();

/* Example 2a: Define a "plain old" class (without using the classify function) */
var Class = function()
{
	this.name = "John";
};

Class.prototype.count = function()
{
	return this.name + ": One. Two. Three.";
};

/* Example 2b: Define a derived class that extends a "plain old" base class */
var SpanishClass = classify(Class, {
	constructor: function()
	{
		this.name = "Juan";
	},
	count: function()
	{
		return this.name + ": Uno. Dos. Tres.";
	}
});

/* Example 3: Define a Person class that extends Object by default */
var Person = classify({
	constructor: function(name, isQuiet)
	{
		this.name = name;
		this.isQuiet = isQuiet;
	},
	canSing: function()
	{
		return !this.isQuiet;
	},
	sing: function()
	{
		return this.canSing() ? "Figaro!" : "Shh!";
	},
	toString: function()
	{
		return "Hello, " + this.name + "!";
	}
});

/* Example 4: Define a Ninja class that extends Person */
var Ninja = classify(Person, {
	constructor: function(name, skillLevel)
	{
		Ninja.base.constructor.call(this, name, true);
		this.skillLevel = skillLevel;
	},
	canSing: function()
	{
		return Ninja.base.canSing.call(this) || this.skillLevel > 200;
	},
	attack: function()
	{
		return "Chop!";
	}
});

/* Example 4: Define an ExtremeNinja class that extends Ninja that extends Person */
var ExtremeNinja = classify(Ninja, {
	attack: function()
	{
		return "Chop! Chop!";
	},
	backflip: function()
	{
		this.skillLevel++;
		return "Woosh!";
	}
});

var m = new Minimal();
var c = new Class();
var s = new SpanishClass();
var p = new Person("Mary", false);
var n = new Ninja("John", 100);
var e = new ExtremeNinja("World", 200);

