﻿using System;
using System.Web.Mvc;
using MvcMultiButtonSampleApp.Models;

namespace MvcMultiButtonSampleApp.Controllers
{
    public class EmployeeBonusController : Controller
    {
        [HttpGet]
        public ActionResult Index()
        {
            var model = ModelFactory.GetInitialCompanyModel();

            return View(model);
        }

        [ButtonHandler()]
        public ActionResult UpdateCompanyBudget(Company model, decimal value)
        {
            // Increase the bonus budget by lowering the benefits of the shareholders:
            model.ShareHoldersBenefit -= value;
            model.Budget += value;

            return ValidatedView(model);
        }

        [ButtonHandler(ButtonName = "AddDepartment")]
        public ActionResult AddDepartment(Company model)
        {
            // Add a new department:
            model.Departments.Add(new Department());

            return ValidatedView(model);
        }

        [ButtonHandler(ArgumentNames = "departmentId")]
        public ActionResult UpdateDepartmentBudget(Company model, int departmentId, decimal value)
        {
            // Increase the departments budget by lowering the companies budget:
            model.Budget -= value;
            model.Departments[departmentId].Budget += value;

            return ValidatedView(model);
        }

        [ButtonHandler(ArgumentNames = "departmentId")]
        public ActionResult DeleteDepartment(Company model, int departmentId)
        {
            // Delete the given department:
            model.Departments.RemoveAt(departmentId);

            return ValidatedView(model);
        }

        [ButtonHandler(ArgumentNames = "departmentId")]
        public ActionResult AddEmployee(Company model, int departmentId)
        {
            // Add a new employee:
            model.Departments[departmentId].Employees.Add(new Employee());

            return ValidatedView(model);
        }

        [ButtonHandler(ArgumentNames = "departmentId, employeeId")]
        public ActionResult UpdateBonus(Company model, int departmentId, int employeeId, decimal value)
        {
            // Increase the bonus of the employee by lowering his departments budget:
            model.Departments[departmentId].Budget -= value;
            model.Departments[departmentId].Employees[employeeId].Bonus += value;

            return ValidatedView(model);
        }

        [ButtonHandler(ArgumentNames = "departmentId, employeeId")]
        public ActionResult DeleteEmployee(Company model, int departmentId, int employeeId)
        {
            // Delete the given employee:
            model.Departments[departmentId].Employees.RemoveAt(employeeId);

            return ValidatedView(model);
        }

        #region Validation

        internal ActionResult ValidatedView(Company model)
        {
            ModelState.Clear();
            if (model.ShareHoldersBenefit < 0m)
                ModelState.AddModelError("ShareHoldersBenefit", "ShareHoldersBenefit cannot be negative.");
            if (model.Budget < 0m)
                ModelState.AddModelError("Budget", "Company budget cannot be negative.");
            for (int d = 0; d < model.Departments.Count; d++)
            {
                var dep = model.Departments[d];
                if (dep.Budget < 0m)
                    ModelState.AddModelError("Departments["+d+"].Budget", "Budget of department '" + dep.Name + "' cannot be negative.");
                for (var e = 0; e < dep.Employees.Count; e++)
                {
                    var emp = dep.Employees[e];
                    if (emp.Bonus < 0m)
                        ModelState.AddModelError("Departments[" + d + "].Employees["+e+"].Bonus", "Bonus of '" + emp.Name + "' cannot be negative.");
                }
            }
            return View(model);
        }

        #endregion
    }
}
