#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.Collections.Generic;
using SharpDom;
using NUnit.Framework;
using SharpDom.Attributes;

namespace SharpDom.UnitTests
{
    [TestFixture]
    public class AttributeTests
    {
        private bool _isInitialized = false;
        private readonly List<Type> _attributeClasses = new List<Type>();

        [SetUp]
        public void Init()
        {
            if (_isInitialized) return;
            var types = typeof (BaseAttr).Assembly.GetTypes();
            foreach (var type in types)
            {
                if (!type.IsSubclassOf(typeof(BaseAttr))) continue;
                if (type == typeof(EnumAttr<>) || type == typeof(EventAttr)) continue;

                _attributeClasses.Add(type);
            }
            _isInitialized = true;
        }

        [Test]
        public void Test_AttributeClassName_StartsWith_PredefinedPrefix()
        {
            foreach (var attributeClass in _attributeClasses)
            {
                Assert.That(attributeClass.Name.StartsWith(Constants.AttributeClassPrefix), Is.True,
                    string.Format("Attribute class '{0}' doesn't start with prefix '{1}'",
                    attributeClass.Name, Constants.AttributeClassPrefix));
            }
        }

        [Test]
        public void Test_ClassName_IsEqualTo_AttributeName()
        {
            foreach (var attributeClass in _attributeClasses)
            {
                var constructor = attributeClass.GetConstructor(Type.EmptyTypes);
                var attribute = (BaseAttr)constructor.Invoke(null);
                var className = attributeClass.Name.Substring(Constants.AttributeClassPrefix.Length);
                var attributeName = attribute.Name;

                if (attributeClass == typeof(AttrCssClass)) className = "Class";

                Assert.That(className.ToLower() == attributeName, Is.True,
                    string.Format("Attribute class '{0}' has incorrect name of attribute '{1}'",
                    className, attributeName));
            }
        }

        [Test]
        public void Test_ScriptAttribute_IsDerivatedFrom_CorrectSubclass()
        {
            foreach (var attributeClass in _attributeClasses)
            {
                if (!attributeClass.Name.StartsWith(Constants.EventAttributeClassPrefix))
                {
                    Assert.That(attributeClass.IsSubclassOf(typeof(EventAttr)), Is.False,
                        string.Format("Attribute class '{0}' starts with prefix '{1}' but it is not event attribute",
                        attributeClass.Name, Constants.EventAttributeClassPrefix));
                }
                else
                {
                    Assert.That(attributeClass.IsSubclassOf(typeof(EventAttr)), Is.True,
                        string.Format("Attribute class '{0}' doesn't starts with prefix '{1}' but it is event attribute",
                        attributeClass.Name, Constants.EventAttributeClassPrefix));
                }
            }
        }
    }
}