﻿#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Xml.Linq;

namespace TypesFromXsdGenerator
{
    class Program
    {
        static void Main(string[] args)
        {
            

            XDocument doc = XDocument.Load("XHTML1Strict.xsd");
            Group<TypeTemplate> types = GetAllTypes(doc);
            Group<AttributeItem> attrGroups = GetAttrGroups(doc, types);
            Group<TagTemplate> tagsGroups = GetTagsGroups(doc);
            List<TagTemplate> tags = GetTags(doc, attrGroups, types, tagsGroups);
            List<RelationTemplate> relations = GetRelations(tags);
            var templates = new List<BuilderTemplate>();
            templates.Add(new BuilderTemplate(tags));

            var adoptTemplates = new List<FullAdoptabilityTemplate>();
            adoptTemplates.Add(new FullAdoptabilityTemplate(tags));
            
            SaveTemplates(types.GatAllItems(),"types");
            SaveTemplates(tags,"tags");
            SaveTemplates(relations,"relations");
            SaveTemplates(templates,"builder");
            SaveTemplates(adoptTemplates, "full adoptability");
        }

        private static void SaveTemplates<T>(List<T> templates, String directory) where T:TemplateBase
        {
            if (Directory.Exists(directory))
            {
                Directory.Delete(directory, true);
                Thread.Sleep(1000);
            }
            Directory.CreateDirectory(directory);
            foreach (var template in templates)
            {
                var tw = new StreamWriter(directory + "\\" + template.GetFileName());

                // write a line of text to the file
                tw.Write(template.Render());

                // close the stream
                tw.Close();
            }
        }

        private static List<RelationTemplate> GetRelations(List<TagTemplate> tags)
        {
            var res = new List<RelationTemplate>();
            foreach (var tag in tags)
            {
                var possibleParents = new List<TagTemplate>();
                for (int i = 0; i < tags.Count; i++)
                {
                    if (tags[i].PossibleChilds.Count(x=>tag.Name == x.Name) >0) possibleParents.Add(tags[i]);
                }
                var item = new RelationTemplate(possibleParents, tag);
                res.Add(item);
            }
            return res;
        }

        private static List<TagTemplate> GetTags(XDocument doc, Group<AttributeItem> attrGroups, Group<TypeTemplate> types, Group<TagTemplate> relationsGroup)
        {
            var res = new Group<TagTemplate>();
            foreach (var el in doc.Root.Elements("{http://www.w3.org/2001/XMLSchema}element"))
            {
                var item = new TagTemplate(el.Attribute("name").Value);
                AddAttributes(el, types, item, attrGroups);
                AddPossibleChilds(el, item, relationsGroup);

                res.Items.Add(item);
            }
            return res.Items;
        }

        private static void AddPossibleChilds(XElement el, TagTemplate item, Group<TagTemplate> relationsGroup)
        {
            foreach (var element in el.Descendants("{http://www.w3.org/2001/XMLSchema}element"))
            {
                var attr = new TagTemplate();
                attr.Name = element.Attribute("ref").Value;
                item.PossibleChilds.Add(attr);
            }
            foreach (var subGroupEleemnt in el.Descendants("{http://www.w3.org/2001/XMLSchema}group"))
            {
                var sbgrName = subGroupEleemnt.Attribute("ref").Value;
                item.PossibleChilds.AddRange(relationsGroup.GetGroupByName(sbgrName).GatAllItems());
            }
        }

        private static void AddAttributes(XElement el, Group<TypeTemplate> types, TagTemplate item, Group<AttributeItem> attrGroups)
        {
            foreach (var attrEleemnt in el.Descendants("{http://www.w3.org/2001/XMLSchema}attribute"))
            {
                var attr = new AttributeItem();
                attr.Populate(attrEleemnt, types);
                item.Attributes.Add(attr);
            }
            foreach (var subGroupEleemnt in el.Descendants("{http://www.w3.org/2001/XMLSchema}attributeGroup"))
            {
                var sbgrName = subGroupEleemnt.Attribute("ref").Value;
                item.Attributes.AddRange(attrGroups.GetGroupByName(sbgrName).GatAllItems());
            }
        }

        private static Group<TagTemplate> GetTagsGroups(XDocument doc)
        {
            var res = new Group<TagTemplate>();
            var groups = new List<XElement>();
            groups.AddRange(doc.Root.Elements("{http://www.w3.org/2001/XMLSchema}complexType"));
            groups.AddRange(doc.Root.Elements("{http://www.w3.org/2001/XMLSchema}group"));
            foreach (var el in groups)
            {
                var item = new Group<TagTemplate>();
                item.Name = el.Attribute("name").Value;
                foreach (var subGroupEleemnt in el.Descendants("{http://www.w3.org/2001/XMLSchema}group"))
                {
                    var sbgrName = subGroupEleemnt.Attribute("ref").Value;
                    item.SubGroups.Add(res.GetGroupByName(sbgrName));
                }
                foreach (var element in doc.Root.Elements("{http://www.w3.org/2001/XMLSchema}element"))
                {
                    var tag = new TagTemplate(element.Attribute("name").Value);
                    res.Items.Add(tag);
                }
                res.SubGroups.Add(item);

            }

            return res;
        }

        private static Group<AttributeItem> GetAttrGroups(XDocument doc, Group<TypeTemplate> types)
        {
            var res = new Group<AttributeItem>();
            foreach (var el in doc.Root.Elements("{http://www.w3.org/2001/XMLSchema}attributeGroup"))
            {
                var item = new Group<AttributeItem>();
                item.Name = el.Attribute("name").Value;
                foreach (var subGroupEleemnt in el.Descendants("{http://www.w3.org/2001/XMLSchema}attributeGroup"))
                {
                    var sbgrName = subGroupEleemnt.Attribute("ref").Value;
                    item.SubGroups.Add(res.GetGroupByName(sbgrName));
                }

                foreach (var attrEleemnt in el.Descendants("{http://www.w3.org/2001/XMLSchema}attribute"))
                {
                    var attr = new AttributeItem();
                    attr.Populate(attrEleemnt, types);
                    item.Items.Add(attr);
                }

                res.SubGroups.Add(item);
            }
            return res;
        }

        

        private static Group<TypeTemplate> GetAllTypes(XDocument doc)
        {
            var res = new Group<TypeTemplate>();
            foreach (var el in doc.Root.Elements("{http://www.w3.org/2001/XMLSchema}simpleType"))
            {
                var item = new TypeTemplate();
                item.Populate(el);
                res.Items.Add(item);
            }
            return res;
        }


    }
}
