﻿#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace SharpDom
{
    /// <summary>Safe collection of tag objects. "Safety" - from the point of view of tag hierarchy.
    /// So adding a tag of another list to this list will automatically unbind the tag from that list.</summary>
    public class TagList : IList<TreeLeafTag>
    {
        /// <summary>Tag owning the list of children tags in tag hierarchy</summary>
        protected TreeNodeTag Owner;
        /// <summary>Internal list of tags</summary>
        protected List<TreeLeafTag> List = new List<TreeLeafTag>();

        public TagList(TreeNodeTag owner)
        {
            Owner = owner;
        }

        /// <summary>Takes part in HTML generation</summary>
        /// <param name="writer">Destionation of rendering</param>
        public virtual void Render(TextWriter writer)
        {
            List.ForEach(tag => tag.Render(writer));
        }

        // --- methods needed to be secured ---

        public virtual void Add(TreeLeafTag item)
        {
            if (List.Contains(item)) return;
            List.Add(item);
            Adopt(item);
        }
        public virtual void Insert(int index, TreeLeafTag item)
        {
            if (List.Contains(item)) return;
            Adopt(item);
            List.Insert(index, item);
        }
        public virtual TreeLeafTag this[int index]
        {
            get { return List[index]; }
            set
            {
                List[index] = value;
                Adopt(value);
            }
        }
        public virtual bool Remove(TreeLeafTag item)
        {
            Free(item);
            return List.Remove(item);
        }
        public virtual void RemoveAt(int index)
        {
            Free(List[index]);
            List.RemoveAt(index);
        }
        public virtual void Clear()
        {
            foreach (var t in List)
            {
                Free(t);
            }
            List.Clear();
        }

        // --- just aggregation of some base methods ---

        public virtual IEnumerator<TreeLeafTag> GetEnumerator() { return List.GetEnumerator(); }
        IEnumerator IEnumerable.GetEnumerator() { return ((IEnumerable)List).GetEnumerator(); }
        public virtual bool Contains(TreeLeafTag item) { return List.Contains(item); }
        public virtual void CopyTo(TreeLeafTag[] array, int arrayIndex) { List.CopyTo(array, arrayIndex); }
        public virtual int Count { get { return List.Count; } }
        public virtual bool IsReadOnly { get { return false; } }
        public virtual int IndexOf(TreeLeafTag item) { return List.IndexOf(item); }

        // --- helpers ---

        protected virtual void Adopt(TreeLeafTag item)
        {
            if (item.Parent == Owner) return;

            if (item.Parent != null)
            {
                if (item.Parent.Children.Contains(item))
                {
                    item.Parent.Children.Remove(item);
                }
            }
            item.Parent = Owner;
        }
        protected virtual void Free(TreeLeafTag item)
        {
            item.Parent = null;
        }
    }
}
