#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System.IO;
using System.Text;

namespace SharpDom
{
    /// <summary>Base attribute class.</summary>
    /// <remarks>Can store name and value of attribute in textual format. 
    /// Also con build HTML piece basing on the attribute state.</remarks>
    public abstract class BaseAttr
    {
        /// <summary>Default constructor saving name of the attribute</summary>
        /// <param name="name">Name of the attribute</param>
        protected BaseAttr(string name)
        {
            Name = name;
            Value = string.Empty;
            IsVisible = true;
        }
        /// <summary>Name of the attribute</summary>
        public virtual string Name { get; protected set; }

        /// <summary>Value of attribute</summary>
        public virtual string Value { get; set; }
        /// <summary>Determines whether the attribute is visible</summary>
        public virtual bool IsVisible { get; set; }

        /// <summary>Takes part imn HTML generation</summary>
        /// <param name="writer">Destionation of rendering</param>
        public virtual void Render(TextWriter writer)
        {
            if (!IsVisible) return;
            writer.Write("{0}=\"{1}\"", Name, Value);
        }
    }

    public abstract class EnumAttr<TEnum> : BaseAttr
        where TEnum : struct
    {
        /// <summary>Value in enum</summary>
        private TEnum _enumValue;

        /// <summary>Default constructor saving the name of attribute</summary>
        /// <param name="name">Name of attribute</param>
        protected EnumAttr(string name) : base(name) { }

        /// <summary>Value of attribute</summary>
        public new TEnum Value
        {
            get { return _enumValue; }
            protected set { _enumValue = value; base.Value = value.ToString();  }
        }
    }

    /// <summary>Derivated attribute class representing JavaScript events attached to the tag</summary>
    public abstract class EventAttr : BaseAttr
    {
        /// <summary>Default constructor forwarding to base constructor</summary>
        /// <param name="name">Name of the attribute</param>
        protected EventAttr(string name) : base(name)
        {
        }
    }
}