﻿#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Text;

namespace SharpDom
{
    /// <summary>Represents a leaf tag in the tag tree.
    /// In other words, the tag can't contain nested tags.</summary>
    /// <example>BR tag, literal tag, etc</example>
    public abstract class TreeLeafTag
    {
        /// <summary>Creates new tag out of hierarchy</summary>
        /// <param name="name">Name of the tag</param>
        protected TreeLeafTag(string name)
        {
            TagName = name;
            Attributes = new Dictionary<Type, BaseAttr>();
            CustomAttributes = new Dictionary<string, string>(StringComparer.Ordinal);
            IsVisible = true;
        }

        /// <summary>Name of the tag, ex. "span"</summary>
        public virtual string TagName { get; protected set; }

        /// <summary>Reference to parent tag in tag hierarchy</summary>
        protected TreeNodeTag _parent;

        /// <summary>Parent of this tag in tag hierarchy</summary>
        public virtual TreeNodeTag Parent
        {
            get { return _parent; }
            set
            {
                if (_parent != null && _parent.Children.Contains(this))
                {
                    _parent.Children.Remove(this);
                }
                if (value != null)
                {
                    value.Children.Add(this);
                }
                _parent = value;
            }
        }

        /// <summary>Attributes of the tag</summary>
        public virtual Dictionary<Type, BaseAttr> Attributes { get; protected set; }

        /// <summary>Custom attributes of the tag</summary>
        public virtual Dictionary<string, string> CustomAttributes { get; protected set; }

        /// <summary>Determines whether the tag is visible (generated to HTML)</summary>
        public virtual bool IsVisible { get; set; }

        #region Render

        /// <summary>Determines accumulated indent size of the current tag on rendering stage</summary>
        public virtual int? IndentSize
        {
            get { return (Parent != null && Parent.IndentSize != null ? Parent.IndentSize + 4 : 0); }
        }

        /// <summary>Renders indent in front of the tag being rendered</summary>
        /// <param name="writer">Writer</param>
        public virtual void RenderBeforeTag(TextWriter writer)
        {
            writer.Write("{0," + IndentSize + "}", string.Empty);
        }

        /// <summary>Takes part in HTML generation</summary>
        /// <param name="writer">Destionation of rendering</param>
        public virtual void Render(TextWriter writer)
        {
            if (!IsVisible) return;

            FireOnPreRender();
            RenderBeforeTag(writer);
            writer.Write("<{0}", TagName);
            foreach (var attribute in Attributes)
            {
                writer.Write(" ");
                attribute.Value.Render(writer);
            }
            foreach (var attribute in CustomAttributes)
            {
                writer.Write(" ");
                writer.Write("{0}=\"{1}\"", attribute.Key, attribute.Value);
            }
            writer.Write(" />");
            RenderAfterTag(writer);
            FireOnPostRender();
        }

        /// <summary>Renders new line after of the tag being rendered</summary>
        /// <param name="writer">Writer</param>
        public virtual void RenderAfterTag(TextWriter writer)
        {
            writer.WriteLine();
        }

        #endregion

        #region Maintenability

        /// <summary>Any purpose label attached on any stage to the tag</summary>
        public virtual object TagLabel { get; set; }

        /// <summary>Internal ID of the tag - contains IDs of all parents.
        /// Each section is just a order number of tag in parent's collection.</summary>
        public virtual int[] InternalId
        {
            get
            {
                var id = new List<int>();
                if (Parent != null)
                {
                    id.AddRange(Parent.InternalId);
                    id.Add(Parent.Children.IndexOf(this));
                }
                return id.ToArray();
            }
        }

        /// <summary>Delegate of event happened against the tag</summary>
        /// <param name="sourceTag">Tag firing the event</param>
        /// <param name="reasonTag">Tag on which event initially happens</param>
        public delegate void TagEventDelegate(TreeLeafTag sourceTag, TreeLeafTag reasonTag);

        /// <summary>Event when the tag is being rendered</summary>
        public event TagEventDelegate OnPreRender;
        protected virtual void FireOnPreRender(TreeLeafTag tag = null)
        {
            if (OnPreRender != null) OnPreRender(this, tag ?? this);
            if (Parent != null) Parent.FireOnPreRender(tag ?? this);
        }

        /// <summary>Event when the tag is rendered</summary>
        public event TagEventDelegate OnPostRender;
        protected virtual void FireOnPostRender(TreeLeafTag tag = null)
        {
            if (OnPostRender != null) OnPostRender(this, tag ?? this);
            if (Parent != null) Parent.FireOnPostRender(tag ?? this);
        }

        #endregion

        #region Helpers

        /// <summary>Helper method to get tag's attribute by given attribute type</summary>
        /// <typeparam name="TAttribute">Type of tag's attribute</typeparam>
        /// <returns>Tag's attribute</returns>
        protected virtual TAttribute GetAttr<TAttribute>() where TAttribute : BaseAttr
        {
            if (!Attributes.ContainsKey(typeof(TAttribute))) return null;
            return (TAttribute)Attributes[typeof(TAttribute)];
        }

        /// <summary>Attach attribute to the tag</summary>
        /// <typeparam name="TAttribute">Type of the attribuet to attach</typeparam>
        /// <param name="attribute">Attribute to attach</param>
        protected virtual void SetAttr<TAttribute>(TAttribute attribute) where TAttribute : BaseAttr
        {
            if (GetAttr<TAttribute>() != null) Attributes.Remove(typeof(TAttribute));
            if (attribute != null) Attributes[typeof(TAttribute)] = attribute;
        }

        /// <summary>Converts .NET String object into literal tag</summary>
        /// <param name="text">Text to convert</param>
        /// <returns>Literal tag</returns>
        public static implicit operator TreeLeafTag(string text)
        {
            return new LiteralTag { Text = text };
        }

        #endregion
    }

    /// <summary>Represents a node tag in the tag tree.
    /// In other words, the tag can contain nested tags.</summary>
    /// <example>HTML tag, DIV tag, etc</example>
    public abstract class TreeNodeTag : TreeLeafTag
    {
        /// <summary>Default constructor, creates empty Children list and calls base constructor</summary>
        /// <param name="name">Name of the tag</param>
        protected TreeNodeTag(string name) : base(name)
        {
            Children = new TagList(this);
        }

        /// <summary>Children tags in tag hierarchy</summary>
        public virtual TagList Children { get; protected set; }

        #region Render

        /// <summary>Takes part in HTML generation</summary>
        /// <param name="writer">Destionation of rendering</param>
        public override void Render(TextWriter writer)
        {
            if (!IsVisible) return;

            FireOnPreRender();
            RenderBeforeTag(writer);
            writer.Write("<{0}", TagName);
            foreach (var attribute in Attributes)
            {
                writer.Write(" ");
                attribute.Value.Render(writer);
            }
            foreach (var attribute in CustomAttributes)
            {
                writer.Write(" ");
                writer.Write("{0}=\"{1}\"", attribute.Key, attribute.Value);
            }
            if (Children.Count == 0)
            {
                writer.Write(" />");
                RenderAfterTag(writer);
            }
            else
            {
                writer.Write(">");
                RenderAfterTag(writer);

                Children.Render(writer);

                RenderBeforeTag(writer);
                writer.Write("</{0}>", TagName);
                RenderAfterTag(writer);
            }
            FireOnPostRender();
        }

        #endregion
    }

    /// <summary>Base class for all tags with possible nested tags.
    /// This class addes indexer support to its parent superclass.</summary>
    public abstract class IndexedTag : TreeNodeTag
    {
        /// <summary>Calls base class supporting tag name</summary>
        /// <param name="name">Name of the tag</param>
        protected IndexedTag(string name) : base(name)
        {
        }

        /// <summary>Indexer accepting an array of nested tags</summary>
        /// <param name="items">Child tags</param>
        /// <returns>This instance</returns>
        public virtual TreeNodeTag this[params TreeLeafTag[] items]
        {
            get
            {
                foreach (var item in items)
                {
                    Children.Add(item);
                }
                return this;
            }
        }

        /// <summary>Adds custom attribute to the tag</summary>
        /// <param name="attrName">Name of the custom attribute</param>
        /// <param name="attrValue">Value of the custom attribute</param>
        /// <returns>This instance</returns>
        public virtual IndexedTag ext(string attrName, string attrValue)
        {
            if (!string.IsNullOrEmpty(attrName))
            {
                CustomAttributes[attrName] = attrValue;
            }
            return this;
        }
    }
}
