#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.Collections.Generic;
using System.IO;
using SharpDom;

namespace ConsoleSample
{
    /// <summary>It is very complex and mixed example of everything</summary>
    public class Sample7GreatMix
    {
        public static void Run()
        {
            // initially we prepare view model
            var contentPageViewModel = new ContentPageViewModel
            {
                Title = "Content Page Title",
                LoginControlViewModel = new LoginControlViewModel
                {
                    FirstText = "Fist Text",
                    SecondText = "Second Text"
                },
                MasterPageViewModel = new MasterPageViewModel
                {
                    Title = "Test Title",
                    UlItems = new List<MasterPageViewModel.ViewModelUlItem>(new[]
                    {
                        new MasterPageViewModel.ViewModelUlItem { SomeNumber = 0, SomeText = "Test Text 1" },
                        new MasterPageViewModel.ViewModelUlItem { SomeNumber = 1, SomeText = "Test Text 2" },
                    })
                }
            };

            // then we create page builder
            var contentPageBuilder = new ContentPageBuilder();
            var writer = new StringWriter();

            // then we supply the page builder with the model
            contentPageBuilder.Render(contentPageViewModel, writer);

            // finally we show generated HTML code
            Console.WriteLine(writer.ToString());
            Console.ReadKey();
        }

        // ----- Model and Builder of the Master Page -----

        public class MasterPageViewModel
        {
            public class ViewModelUlItem
            {
                public string SomeText { get; set; }
                public int SomeNumber { get; set; }
            }

            public string Title { get; set; }
            public List<ViewModelUlItem> UlItems;
        }

        public class MasterPageBuilder : HtmlPageBuilder<MasterPageViewModel>
        {
            public virtual PlaceHolderDelegate FirstPlaceHolder { get; set; }
            public virtual PlaceHolderDelegate SecondPlaceHolder { get; set; }

            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return container
                [
                    html[
                        head[
                            title[ Model.Title ],
                            script.attr(charset: "Latin1")
                        ],
                        body.attr(onunload: "confirm();", onload: "alert();")[
                            div.attr(style: "display:none;", id: "id1", cssclass: "hidden"),
                            div[ "test" ],
                            div.attr(style: "display:none;")[
                                div[
                                    span.attr(style: "test")["test"],
                                    ul[
                                        ForEach(Model.UlItems, (item, c) => c
                                        [
                                            If(item.SomeNumber == 0) [ li["First item" ] ],
                                            li.attr(cssclass: item.SomeText)
                                        ]),
                                        li.attr(lang: "English", id: "test")
                                    ]
                                ]
                            ],
                            PlaceHolder(FirstPlaceHolder),
                            span[ "some text" ],
                            PlaceHolder(SecondPlaceHolder, c => c[ span[ "stub for first placeholder" ] ])
                        ]
                    ]
                ];
            }
        }

        // ----- Model and Builder of the Content Page -----

        public class ContentPageViewModel
        {
            public string Title;
            public LoginControlViewModel LoginControlViewModel;
            public MasterPageViewModel MasterPageViewModel;
        }

        public class ContentPageBuilder : HtmlPageBuilder<ContentPageViewModel>
        {
            public virtual LoginControl LoginControl { get { return CreateBuilder<LoginControl>(); } }

            public virtual ContainerTag FirstPlaceHolderContent(ContainerTag container)
            {
                return container
                [
                    span[ "start!" ],
                    div[ Model.Title ],
                    LoginControl.Attach<LoginControl>(b => b.SomePlaceHolder = c => c[
                        span[ "overriden content of the user control's placeholder" ]
                    ]).CreateTags(Model.LoginControlViewModel)
                ];
            }

            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return CreateBuilder<MasterPageBuilder>(container)
                    .Attach<MasterPageBuilder>(b => b.FirstPlaceHolder = FirstPlaceHolderContent)
                    .CreateTags(Model.MasterPageViewModel);
            }
        }

        // ----- Model and Builder of the User Control -----

        public struct LoginControlViewModel
        {
            public string FirstText { get; set; }
            public string SecondText { get; set; }
        }

        public class LoginControl : UserControlBuilder<LoginControlViewModel>
        {
            public virtual PlaceHolderDelegate SomePlaceHolder { get; set; }
            public virtual SubControl subControl { get { return CreateBuilder<SubControl>(); } }

            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return container
                [
                    div[ Model.FirstText ],
                    PlaceHolder(SomePlaceHolder, c => c[
                        span[ "some default content of the placeholder" ]
                    ]),
                    span[ Model.SecondText ],
                    subControl.CreateTags(Model)
                ];
            }
        }

        /// <summary>Example of subcontrol nested in other control</summary>
        public class SubControl : UserControlBuilder<LoginControlViewModel>
        {
            /// <summary>Creates some HTML tags as a content of subcontrol</summary>
            /// <returns>Boundary containg all tags</returns>
            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return container
                [
                    span[ Model.FirstText ],
                    span[ "TEST OF SUBCONTROL" ]
                ];
            }
        }
    }
}