#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.IO;
using SharpDom;

namespace ConsoleSample
{
    /// <summary>
    /// This sample demonstrates how to design and to use user controls.
    /// 1. Initially, during design, the user control should derivate from UserControlBuilder class
    /// and as usual override CreateTagTree method in order to supply needed HTML layout.
    /// Sure, the user control has its own view model which can be used for customization.
    /// 2. On using, page builder should have property of the user control's class
    /// properly initialized and then this property can be used in HTML layout directly.
    /// 3. User controls could have nested user controls in the same manner
    /// as pages could have nested user controls (see such example in following samples).
    /// </summary>
    public class Sample4UserControls
    {
        public static void Run()
        {
            // initially we prepare view model
            // notice - we create not only view model for page builder 
            // but also a view model for user control
            var model = new ViewModel
            {
                Title = "Page Title",
                ControlModel = new ControlModel
                {
                    Text = "Some text inside the control",
                    Value = "value of the control"
                }
            };

            // then we create page builder
            var pageBuilder = new PageBuilder();
            var writer = new StringWriter();

            // then we supply the page builder with the model
            pageBuilder.Render(model, writer);

            // finally we show generated HTML code
            Console.WriteLine(writer.ToString());
            Console.ReadKey();
        }

        // ----- Model and Builder of the Page -----

        /// <summary>Modifed view model contains now additional part - 
        /// view model for the user control</summary>
        public class ViewModel
        {
            public string Title { get; set; }
            public ControlModel ControlModel { get; set; }
        }

        /// <summary>Now HTML layout is not only dynamic, it is partial now.
        /// Some HTML layout is moved into different boundary named "user control"</summary>
        public class PageBuilder : HtmlPageBuilder<ViewModel>
        {
            /// <summary>SEE HERE ! - instance of the user control - should be public and virtual.</summary>
            public virtual ControlBuilder ControlBuilder { get { return CreateBuilder<ControlBuilder>(); } }

            /// <summary>Again, again and again - this methods should be overriden in all page builders.
            /// It should contain real HTML layout expressed in terms of C# language.</summary>
            /// <param name="container">The container inside which new HTML layout should be placed.</param>
            /// <returns>The same container tag as incoming one</returns>
            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return container
                [
                    html[
                        head[
                            title[ Model.Title ]
                        ],
                        body[
                            div[
                                // SEE HERE ! - example of using the user control
                                ControlBuilder.CreateTags(Model.ControlModel)
                            ]
                        ]
                    ]
                ];
            }
        }

        // ----- Model and Builder of the User Control -----

        /// <summary>It is the view model of the user control only</summary>
        public class ControlModel
        {
            public string Text { get; set; }
            public string Value { get; set; }
        }

        /// <summary>SEE HERE ! - user control generates HTML layout of only its own competency</summary>
        public class ControlBuilder : UserControlBuilder<ControlModel>
        {
            /// <summary>This methods should be overriden not only in page builders but also in control builders as well.
            /// It should contain real HTML layout expressed in terms of C# language.</summary>
            /// <param name="container">The container inside which new HTML layout should be placed.</param>
            /// <returns>The same container tag as incoming one</returns>
            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return container
                [
                    div[ "Text is " + Model.Text ],
                    div[ "Value is " + Model.Value ] 
                ];
            }
        }
    }
}