#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.IO;
using SharpDom;

namespace ConsoleSample
{
    /// <summary>
    /// This sample demonstrates how to use operators.
    /// Now ther eare only two operators supported - "Foreach" and "If".
    /// 1. "ForEach" operator duplicates inner HTML tags taking into account
    /// supplied items in IEnumerable collections -  so each duplicate of
    /// HTML piece can be customized using the proper supplied item.
    /// 2. "If" operator just hides inner HTML tags if the condition supplied 
    /// as operand is not True. Otherwise, HTML tags are left as they are.
    /// </summary>
    public class Sample3Operators
    {
        public static void Run()
        {
            // initially we prepare view model
            // notice - we create a list of elements to be supplied to ForEach operator
            var model = new ViewModel
            {
                Title = "Page Title",
                UlItems = new[]
                {
                    new ViewModel.UlItem { Text = "Item 1", Value = 1 },
                    new ViewModel.UlItem { Text = "Item 2", Value = 2 }
                }
            };

            // then we create page builder
            var pageBuilder = new PageBuilder();
            var writer = new StringWriter();

            // then we supply the page builder with the model
            pageBuilder.Render(model, writer);

            // finally we show generated HTML code
            Console.WriteLine(writer.ToString());
            Console.ReadKey();
        }

        // ----- Model and Builder of the Page -----

        /// <summary>Modifed view model contains not only page title,
        /// but also a list of elements to be represented as LI tags inside UL tag</summary>
        public class ViewModel
        {
            public class UlItem
            {
                public string Text { get; set; }
                public int Value { get; set; }
            }
            public string Title { get; set; }
            public UlItem[] UlItems { get; set; }
        }

        /// <summary>Now HTML layout is more dynamic. It has now somewhat of 
        /// ASP.NET WebForms' Repeator and also some conditional branching</summary>
        public class PageBuilder : HtmlPageBuilder<ViewModel>
        {
            /// <summary>Again and again - this methods should be overriden in all page builders.
            /// It should contain real HTML layout expressed in terms of C# language.</summary>
            /// <param name="container">The container inside which new HTML layout should be placed.</param>
            /// <returns>The same container tag as incoming one</returns>
            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return container
                [
                    html[
                        head[
                            title[ Model.Title ]
                        ],
                        body[
                            div[
                                ul[

                                    // SEE HERE ! - this is the example of usage of the "Foreach" operator
                                    ForEach(Model.UlItems, (item, c) => c[
                                        li.attr(id: item.Value.ToString())[ item.Text ]
                                    ])

                                ],

                                // SEE HERE ! - this is the example of usage of the "If" operator
                                If(Model.UlItems.Length % 2 != 0)[
                                    span[ "Amount of items is odd ..." ]
                                ]
                            ]
                        ]
                    ]
                ];
            }
        }
    }

}