window.TodoApp = window.TodoApp || {};

// represents a single todo item
window.TodoApp.Todo = function (id, title, completed) {
    var me = this;
    this.id = ko.observable(id);
    this.title = ko.observable(title);
    this.completed = ko.observable(completed || false);
    this.editing = ko.observable(false);

    // edit an item
    this.startEdit = function () {
        me.editing(true);
    };

    // stop editing an item
    this.stopEdit = function (data, event) {
        if (event.keyCode == 13) {
            me.editing(false);
        }
        return true;
    };
};

// our main view model
window.TodoApp.ViewModel = function (spHostUrl) {
    var me = this;

    this.todos = ko.observableArray(); //List of todos
    this.current = ko.observable(); // store the new todo value being entered
    this.showMode = ko.observable('all'); //Current display mode

    //List which is currently displayed
    this.filteredTodos = ko.computed(function () {
        switch (me.showMode()) {
        case 'active':
            return me.todos().filter(function (todo) {
                return !todo.completed();
            });
        case 'completed':
            return me.todos().filter(function (todo) {
                return todo.completed();
            });
        default:
            return me.todos();
        }
    });

    this.addTodo = function (todo) {
        me.todos.push(todo);
        var adding = true;
        ko.computed(function () {
            var title = todo.title(),
                completed = todo.completed();

            if (!adding) {
                $.ajax({
                    type: 'POST',
                    url: "/Home/UpdateItem?SPHostUrl=" + encodeURIComponent(spHostUrl),
                    contentType: "application/json; charset=utf-8",
                    data: JSON.stringify({
                        id: todo.id(),
                        title: title,
                        completed: completed
                    }),
                    dataType: "json"
                });
            }
        });
        adding = false;
    }

    // add a new todo, when enter key is pressed
    this.add = function (data, event) {
        if (event.keyCode == 13) {
            var current = me.current().trim();
            if (current) {
                var todo = new window.TodoApp.Todo(0, current);
                me.addTodo(todo);
                me.current('');

                $.ajax({
                    type: 'POST',
                    url: "/Home/AddItem?SPHostUrl=" + encodeURIComponent(spHostUrl),
                    contentType: "application/json; charset=utf-8",
                    data: JSON.stringify({
                        title: todo.title()
                    }),
                    dataType: "json",
                    success: function (id) {
                        todo.id(id);
                    }
                });
            }
        }
        return true;
    };

    // remove a single todo
    this.remove = function (todo) {
        $.ajax({
            type: 'POST',
            url: "/Home/RemoveItem?SPHostUrl=" + encodeURIComponent(spHostUrl),
            contentType: "application/json; charset=utf-8",
            data: JSON.stringify({
                id: todo.id()
            }),
            dataType: "json"
        });
        me.todos.remove(todo);
    };

    // remove all completed todos
    this.removeCompleted = function () {
        var todos = me.todos().slice(0);
        for (var i = 0; i < todos.length; i++) {
            if (todos[i].completed()) {
                me.remove(todos[i]);
            }
        }
    };

    // count of all completed todos
    this.completedCount = ko.computed(function () {
        return me.todos().filter(function (todo) {
            return todo.completed();
        }).length;
    });

    // count of todos that are not complete
    this.remainingCount = ko.computed(function () {
        return me.todos().length - me.completedCount();
    });

    // writeable computed observable to handle marking all complete/incomplete
    this.allCompleted = ko.computed({
        //always return true/false based on the done flag of all todos
        read: function () {
            return !me.remainingCount();
        },
        // set all todos to the written value (true/false)
        write: function (newValue) {
            me.todos().forEach(function (todo) {
                // set even if value is the same, as subscribers are not notified in that case
                todo.completed(newValue);
            });
        }
    });
};
