﻿using System;
using System.Collections;
using System.Linq;
using System.Web.Mvc;
using Sandtrap.Web.DataAnnotations;
using System.Text.RegularExpressions;

namespace Sandtrap.Web.Extensions
{

    /// <summary>
    /// 
    /// </summary>
    public static class ModelMetadataExtensions
    {

        /// <summary>
        /// Returns the default value for the property.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        public static object DefaultValue(this ModelMetadata metaData)
        {
            return metaData.ModelType.IsValueType ? Activator.CreateInstance(metaData.ModelType) : null;
        }

        /// <summary>
        /// Returns a value indicating if a property is a collection.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        public static bool IsCollection(this ModelMetadata metaData)
        {
            if (metaData.ModelType == typeof(string))
            {
                return false;
            }
            return typeof(IEnumerable).IsAssignableFrom(metaData.ModelType);
        }

        /// <summary>
        /// Returns a value indicating a property has been decorated with the 
        /// <see cref="System.Web.Mvc.HiddenInputAttribute"/>.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        public static bool IsHidden(this ModelMetadata metaData)
        {
            return metaData.TemplateHint == "HiddenInput";
        }

        #region .Table 

        /// <summary>
        /// Gets a value indicating if rows can be added in a table generated by the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableEditorFor"/> method.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool TableCanAddRows(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableEditAttribute.AllowAdditionsKey);
        }

        /// <summary>
        /// Gets a value indicating if rows can be deleted in a table generated by the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableEditorFor"/> method.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool TableCanDeleteRows(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableEditAttribute.AllowDeletionsKey);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="metaData">
        /// 
        /// </param>
        internal static string TableIsActiveProperty(this ModelMetadata metaData)
        {
            if (metaData.AdditionalValues.ContainsKey(TableEditAttribute.IsActivePropertyKey))
            {
                return (string)metaData.AdditionalValues[TableEditAttribute.IsActivePropertyKey];
            }
            return "IsActive";
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="metaData">
        /// 
        /// </param>
        internal static string TableIsDirtyProperty(this ModelMetadata metaData)
        {
            if (metaData.AdditionalValues.ContainsKey(TableEditAttribute.IsDirtyPropertyKey))
            {
                return (string)metaData.AdditionalValues[TableEditAttribute.IsDirtyPropertyKey];
            }
            return "IsDirty";
        }

        /// <summary>
        /// Return a value indicating if rows in a table generated with the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableEditorFor"/> method should 
        /// include row numbers. 
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool TableIncludeRowNumbers(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableDisplayAttribute.IncludeRowNumbersKey);
        }

        /// <summary>
        /// Returns a value indicating if a table generated with the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableDisplayFor"/> method should 
        /// include a column with a link to an edit action method. 
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool TableIncludeEditLink(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableDisplayAttribute.IncludeEditLinkKey);
        }

        /// <summary>
        /// Returns a value indicating if a table generated with the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableDisplayFor"/> method should 
        /// include a column with a link to a details action method. 
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool TableIncludeViewLink(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableDisplayAttribute.IncludeViewLinkKey);
        }



        internal static bool TableIsLink(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableLinkAttribute.TableLinkKey);
        }


        #endregion

        #region .Table column 

        /// <summary>
        /// Returns a value indicating if a property decorated with the 
        /// <see cref="Sandtrap.Web.DataAnnotations.TableColumnAttribute"/> is excluded from 
        /// tables generated by the <see cref="Sandtrap.Web.Html.TableHelper.TableDisplayFor"/>
        /// and <see cref="Sandtrap.Web.Html.TableHelper.TableEditorFor"/> methods.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool ColumnIsExcluded(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableColumnAttribute.ExcludeKey);
        }

        /// <summary>
        /// Returns a value indicating if a property decorated with the 
        /// <see cref="Sandtrap.Web.DataAnnotations.TableColumnAttribute"/> is readonly in a 
        /// table generated by the <see cref="Sandtrap.Web.Html.TableHelper.TableEditorFor"/> 
        /// method.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool ColumnIsReadOnly(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableColumnAttribute.IsReadOnlyKey);
        }

        /// <summary>
        /// Returns a value indicating if a property decorated with the 
        /// <see cref="Sandtrap.Web.DataAnnotations.TableColumnAttribute"/> 
        /// has a display property.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool ColumnHasDisplayProperty(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableColumnAttribute.DisplayPropertyKey);
        }

        /// <summary>
        /// Returns a value indicating if the value of a property decorated with the 
        /// <see cref="Sandtrap.Web.DataAnnotations.TableColumnAttribute"/> should
        /// be ommited if the previous row contains the same value.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        /// <remarks>
        /// If true, and the preceeding row contains the same value, no text is rendered
        /// in the table cell.
        /// </remarks>
        internal static bool ColumnNoRepeat(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableColumnAttribute.NoRepeatKey);
        }

        /// <summary>
        /// Returns a value indicating if a property decorated with the 
        /// <see cref="Sandtrap.Web.DataAnnotations.TableColumnAttribute"/> 
        /// is totalled in the footers of tables generated by the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableDisplayFor"/>
        /// and <see cref="Sandtrap.Web.Html.TableHelper.TableEditorFor"/> methods.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool ColumnIncludeTotals(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableColumnAttribute.IncludeTotalKey);
        }

        /// <summary>
        /// Returns a value indicating if a property decorated with the 
        /// <see cref="Sandtrap.Web.DataAnnotations.TableColumnAttribute"/> 
        /// should be rendered as a hyperlink in tables generated by the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableDisplayFor"/>
        /// and <see cref="Sandtrap.Web.Html.TableHelper.TableEditorFor"/> methods.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool ColumnIsLink(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableLinkAttribute.TableLinkKey);
        }

        /// <summary>
        /// Returns a value indicating if a property decorated with the 
        /// <see cref="Sandtrap.Web.DataAnnotations.TableSelectAttribute"/> 
        /// should be rendered as a select control in tables generated by the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableEditorFor"/> method.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        internal static bool ColumnIsSelect(this ModelMetadata metaData)
        {
            return metaData.AdditionalValues.ContainsKey(TableSelectAttribute.TableSelectKey);
        }

        #endregion

        /// <summary>
        /// Returns a value indicating if the metadata model is a boolean value.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        public static bool IsBoolean(this ModelMetadata metaData)
        {
            return metaData.ModelType == typeof(bool);
        }

        /// <summary>
        /// Returns a value indicating if the metadata model should be displayed as 
        /// currency.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        public static bool IsCurrency(this ModelMetadata metaData)
        {
            if (metaData.DataTypeName == "Currency")
            {
                return true;
            }
            // Check the format string
            Regex regex = new Regex(@"{\d{1}:C\d?}", RegexOptions.IgnoreCase);
            return regex.IsMatch(metaData.DisplayFormatString ?? string.Empty);
        }

        /// <summary>
        /// Returns a value indicating if the metadata model should be displayed as a date.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        public static bool IsDate(this ModelMetadata metaData)
        {
            return metaData.DataTypeName == "Date";
        }

        /// <summary>
        /// Returns a value indicating if the metadata model should be displayed as a percent.
        /// </summary>
        /// <param name="metaData">
        /// The ModelMetadata instance that this method extends.
        /// </param>
        public static bool IsPercent(this ModelMetadata metaData)
        {
            // Why doesn't the framework have a DataTypeName enumeration for Percent?
            if (metaData.TemplateHint == "Percent")
            {
                return true;
            }
            // Check the format string
            Regex regex = new Regex(@"{\d{1}:P\d?}", RegexOptions.IgnoreCase);
            return regex.IsMatch(metaData.DisplayFormatString ?? string.Empty);
        }

    }

}
