﻿using System;
using System.Linq;
using System.Web.Mvc;

namespace Sandtrap.Web.DataAnnotations
{

    /// <summary>
    /// An attribute used to determine if a table generated by the 
    /// <see cref="Sandtrap.Web.Html.TableHelper.TableEditorFor"/> method 
    /// renders the html to allow the dynamic addition and/or deletion of rows.
    /// </summary>
    /// <remarks>
    /// When applied, an additional column with buttons to add and/or delete 
    /// rows is rendered in the table. 
    /// </remarks>
    [AttributeUsage(AttributeTargets.Class)]
    public class TableEditAttribute : Attribute, IMetadataAware
    {

        #region .Declarations 

        // Error messages
        private const string _InvalidProperty = "The class definition '{0}' does not contain a public property named '{1}'.";
        private const string _NotBool = " The property '{0}' is not a boolean.";

        #endregion


        #region .Constructors 

        /// <summary>
        /// Initialises a new instance of TableEditorAttribute class with default properties.
        /// </summary>
        /// <remarks>
        /// The default values for the <see cref="AllowAdditions"/> and <see cref="AllowDeletions"/> 
        /// properties are true. Applying the attribute setting the properties renders the html to 
        /// allow both addition and deletion of rows.
        /// </remarks>
        public TableEditAttribute()
        {
            // Set defaults
            AllowAdditions = true;
            AllowDeletions = true;
        }

        #endregion

        #region .Metadata keys 

        /// <summary>
        /// Gets the key for the metadata <see cref="AllowAdditions"/> property.
        /// </summary>
        public static string AllowAdditionsKey
        {
            get { return "TableAllowAdditions"; }
        }

        /// <summary>
        /// Gets the key for the metadata <see cref="AllowDeletions"/> property.
        /// </summary>
        public static string AllowDeletionsKey
        {
            get { return "TableAllowDeletions"; }
        }

        /// <summary>
        /// Gets the key for the metadata <see cref="IsDirtyProperty"/> property.
        /// </summary>
        public static string IsDirtyPropertyKey
        {
            get { return "TableIsDirtyProperty"; }

        }

        /// <summary>
        /// Gets the key for the metadata <see cref="IsDirtyProperty"/> property.
        /// </summary>
        public static string IsActivePropertyKey
        {
            get { return "TableIsActiveProperty"; }

        }

        #endregion

        #region .Properties 

        /// <summary>
        /// Gets or sets a value indicating if rows can be added to the table.
        /// The default is true.
        /// </summary>   
        public bool AllowAdditions { get; set; }

        /// <summary>
        /// Gets or sets a value indicating if rows can be deleted from the table.
        /// The default is true.
        /// </summary>
        public bool AllowDeletions { get; set; }

        /// <summary>
        /// Gets or sets the name of the property that identifies if the model has changed.
        /// </summary>
        public string IsDirtyProperty { get; set; }

        /// <summary>
        /// Gets or sets the name of the property that identifies if the model is active.
        /// </summary>
        public string IsActiveProperty { get; set; }


        #endregion

        #region .Methods 

        /// <summary>
        /// Adds additional metedata values used to render the html for an editable table.
        /// </summary>
        public void OnMetadataCreated(ModelMetadata metaData)
        {
            if (AllowAdditions)
            {
                metaData.AdditionalValues[AllowAdditionsKey] = true;
            }
            if (AllowDeletions)
            {
                metaData.AdditionalValues[AllowDeletionsKey] = true;
            }
            // Check the IsDirty and IsActive exist and are booleans.
            if (IsDirtyProperty != null)
            {
                ModelMetadata isDirtyMetadata = metaData.Properties
                    .FirstOrDefault(m => m.PropertyName == IsDirtyProperty);
                if (isDirtyMetadata == null)
                {
                    throw new ArgumentNullException(string
                        .Format(_InvalidProperty, metaData.ModelType.Name, IsDirtyProperty));
                }
                if (isDirtyMetadata.ModelType != typeof(bool))
                {
                    throw new ArgumentException(string.Format(_NotBool, IsDirtyProperty));
                }
                metaData.AdditionalValues[IsDirtyPropertyKey] = IsDirtyProperty;
            }
            if (IsActiveProperty != null)
            {
                ModelMetadata isActiveMetadata = metaData.Properties
                    .FirstOrDefault(m => m.PropertyName == IsActiveProperty);
                if (isActiveMetadata == null)
                {
                    throw new ArgumentNullException(string
                        .Format(_InvalidProperty, metaData.ModelType.Name, IsActiveProperty));
                }
                if (isActiveMetadata.ModelType != typeof(bool))
                {
                    throw new ArgumentException(string.Format(_NotBool, IsDirtyProperty));
                }
                metaData.AdditionalValues[IsActivePropertyKey] = IsActiveProperty;
            }  
        }

        #endregion

    }

}
