﻿using System;
using System.Linq;
using System.Web;
using System.Web.Mvc;

namespace Sandtrap.Web.DataAnnotations
{

    /// <summary>
    /// An attribute used to determine display options in table generated by the 
    /// <see cref="Sandtrap.Web.Html.TableHelper.TableDisplayFor"/> method.
    /// </summary>
    /// <exception cref="ArgumentNullException">
    /// If <see cref="ViewAction"/> or <see cref="EditAction"/> is set and 
    /// <see cref="Controller"/> is null.
    /// </exception>
    /// <exception cref="ArgumentException">
    /// <see cref="ViewAction"/> or <see cref="EditAction"/> is set and 
    /// the class does not contain the property identified by <see cref="IDProperty"/>.
    /// </exception>
    /// <remarks>
    /// The attribute contains properties used to determine if additional columns are
    /// rendered to display row numbers, view details hyperlinks and edit hyperlinks.
    /// </remarks>
    [AttributeUsage(AttributeTargets.Class)]
    public class TableDisplayAttribute : Attribute, IMetadataAware
    {

        #region .Declarations 

        // Error messages
        private const string _InvalidController = "The controller cannot be null.";
        private const string _InvalidProperty = "The class definition '{0}' does not contain " + 
             "a public property named '{1}'";

        #endregion

        #region .Constructors 
        
        /// <summary>
        /// Initialises a new instance of TableDisplayAttribute class with default properties.
        /// </summary>
        /// <remarks>
        /// The default value for  <see cref="IncludeRowNumbers"/> is false.
        /// </remarks>
        public TableDisplayAttribute()
        {
            // Set defaults
            IDProperty = "ID";
        }

        #endregion

        #region .Metadata keys 

        /// <summary>
        /// Gets the key for the metadata IncludeViewLink property
        /// </summary>
        /// <remarks>
        /// The key is used to check if a table generated by the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableDisplayFor"/> method
        /// should render an additional column with a hyper link to a details page.
        /// </remarks>
        public static string IncludeViewLinkKey
        {
            get { return "TableIncludeViewLink"; }
        }

        /// <summary>
        /// Gets the key for the metadata IncludeEditLink property.
        /// </summary>
        /// <remarks>
        /// The key is used to check if a table generated by the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableDisplayFor"/> method
        /// should render an additional column with a hyperlink to an edit page.
        /// </remarks>
        public static string IncludeEditLinkKey
        {
            get { return "TableIncludeEditLink"; }
        }

        /// <summary>
        /// Gets the key for the metadata Url property.
        /// </summary>
        /// <remarks>
        /// The key is used to return the Url to an edit page.
        /// The Url is determined by the <see cref="Controller"/>, 
        /// <see cref="EditAction"/> and <see cref="IDProperty"/> properties.
        /// </remarks>
        public static string UrlEditKey
        {
            get { return "TableEditLinkUrl"; }
        }

        /// <summary>
        /// Gets the key for the metadata Url property.
        /// </summary>
        /// <remarks>
        /// The key is used to return the Url to an view page.
        /// The Url is determined by the <see cref="Controller"/>, 
        /// <see cref="ViewAction"/> and <see cref="IDProperty"/> properties.
        /// </remarks>
        public static string UrlViewKey
        {
            get { return "TableViewLinkUrl"; }

        }
   
        /// <summary>
        /// Gets the key for the metadata <see cref="IncludeRowNumbers"/> property.
        /// </summary>
        public static string IncludeRowNumbersKey
        {
            get { return "TableIncludeRowNumbers"; }
        }

        #endregion

        #region .Properties 

        /// <summary>
        /// Gets or sets a value indicating if a table generated by the 
        /// <see cref="Sandtrap.Web.Html.TableHelper.TableDisplayFor"/> method 
        /// renders an additional column to display row numbers.
        /// The default is false.
        /// </summary>   
        public bool IncludeRowNumbers { get; set; }

        /// <summary>
        /// Gets or sets the name of the route controller name used to generate 
        /// view and edit links.
        /// </summary>
        /// <remarks>
        /// The value is ignored if the <see cref="IDProperty"/> and 
        /// <see cref="ViewAction"/> or <see cref="EditAction"/> properties are not set.
        /// </remarks>
        public string Controller { get; set; }

        /// <summary>
        /// Gets or sets the name of the route action method used to view details of 
        /// a row.
        /// </summary>
        /// <remarks>
        /// The value is ignored if the <see cref="Controller"/> and 
        /// <see cref="IDProperty"/> properties are not set.
        /// </remarks>
        public string ViewAction { get; set; }

        /// <summary>
        /// Gets or sets the name of the route action method used to edit details 
        /// of a row.
        /// </summary>
        /// <remarks>
        /// The value is ignored if the <see cref="Controller"/> and 
        /// <see cref="IDProperty"/> properties are not set.
        /// </remarks>
        public string EditAction { get; set; }

        /// <summary>
        /// Gets or sets the name of the property that provides the value to the  
        /// route action method parameter. 
        /// The default is 'ID'.
        /// </summary>
        /// <remarks>
        /// The value is ignored if the <see cref="Controller"/> and 
        /// <see cref="ViewAction"/> or <see cref="EditAction"/> properties are not set.
        /// </remarks>
        public string IDProperty { get; set; }

        #endregion

        #region .Methods 

        /// <summary>
        /// Adds additional metedata values used to render the html for an readonly table.
        /// </summary>
        public void OnMetadataCreated(ModelMetadata metaData)
        {
            if (IncludeRowNumbers)
            {
                metaData.AdditionalValues[IncludeRowNumbersKey] = true;
            }
            if (ViewAction != null || EditAction != null)
            {
                // Check the controller has been provided
                if (Controller == null)
                {
                    throw new ArgumentNullException(_InvalidController);
                }
                // Check the ID property exists
                ModelMetadata idMetaData = metaData.Properties
                    .FirstOrDefault(m => m.PropertyName == IDProperty);
                if (idMetaData == null)
                {
                    throw new ArgumentException(string
                        .Format(_InvalidProperty, metaData.ModelType.Name, IDProperty));
                }
                else
                {
                    idMetaData.AdditionalValues.Add("xxx", "yyy");
                }
                // Build the url's
                string root = HttpRuntime.AppDomainAppVirtualPath;
                if (root == "/")
                {
                    root = string.Empty;
                }
                if (ViewAction != null)
                {
                    // Add metadata
                    metaData.AdditionalValues[IncludeViewLinkKey] = true;
                    if (metaData.Model != null)
                    {
                         metaData.AdditionalValues[UrlViewKey] = string.
                            Format("{0}/{1}/{2}/{3}", root, Controller, ViewAction, idMetaData.Model);
                    }
                }
                if (EditAction != null)
                {
                    // Add metadata
                    metaData.AdditionalValues[IncludeEditLinkKey] = true;
                    if (metaData.Model != null)
                    {
                        metaData.AdditionalValues[UrlEditKey] = string.
                           Format("{0}/{1}/{2}/{3}", root, Controller, EditAction, idMetaData.Model);
                    }
                }
            }
        }

        #endregion
         
    }

}
