<?php

  /**
   * Squiloople Framework
   *
   * LICENSE: Feel free to use and redistribute this code.
   *
   * @author Michael Rushton <michael@squiloople.com>
   * @link http://squiloople.com/
   * @package Squiloople
   * @version 1.0
   * @copyright  2012 Michael Rushton
   */

  /**
   * Password Validator
   *
   * Hash, generate, or validate passwords
   */
  final class PasswordValidator
  {

    /**
     * The password
     *
     * @access private
     * @var string $_password
     */
    private $_password;

    /**
     * The salt
     *
     * @access private
     * @var string $_salt
     */
    private $_salt;

    /**
     * The pepper
     */
    const PEPPER = 'Sz^3X6r[UyvV~2]_0stT}8 uY7RwZx4{q|Q91W5';

    /**
     * Set the password
     *
     * @access public
     * @param string $password
     */
    public function __construct($password = '')
    {
      $this->_password = $password;
    }

    /**
     * Call the constructor fluently
     *
     * @access public
     * @static
     * @param string $password
     * @return PasswordValidator
     */
    public static function setPassword($password)
    {
      return new self($password);
    }

    /**
     * Return the password
     *
     * @access public
     * @return string
     */
    public function getPassword()
    {
      return $this->_password;
    }

    /**
     * Generate a random password
     *
     * @access public
     * @return PasswordValidator
     */
    public function randomizePassword()
    {

      // Randomize the password
      $this->_password = substr($this->getSalt(TRUE), 0, 8);

      // Reset the salt
      $this->_salt = NULL;

      // Return the PasswordValidator object
      return $this;

    }

    /**
     * The password must contain at least one letter of each case, one digit, and be between 8 and 39 characters inclusive in length
     *
     * @access public
     * @param boolean|string $new
     * @return integer
     */
    public function isValid($new = FALSE)
    {
      return preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*[0-9])[\x20-\x7E]{8,39}$/D', $new ?: $this->_password);
    }

    /**
     * Set the salt
     *
     * @access public
     * @param string $salt
     * @return PasswordValidator
     */
    public function setSalt($salt)
    {

      // Set the salt
      $this->_salt = $salt;

      // Return the PasswordValidator object
      return $this;

    }

    /**
     * Generate and return a salt
     *
     * @access public
     * @param boolean $reset
     * @return string
     */
    public function getSalt($reset = FALSE)
    {

      // Return the salt if one has been set and a reset is not required
      if (!$reset && isset($this->_salt))
      {
        return $this->_salt;
      }

      // Generate a random salt of 39 printable ASCII characters
      for ($i = 1; $i <= 39; ++$i)
      {
        $salt[] = chr(mt_rand(32, 126));
      }

      // Return the random salt
      return $this->_salt = implode($salt);

    }

    /**
     * Return the pepper portion
     *
     * @access private
     * @return string
     */
    private function _getPepper()
    {
      return substr(self::PEPPER, 0, 39 - strlen($this->_password));
    }

    /**
     * Hash the password
     *
     * @access private
     * @return string
     */
    private function _getHash()
    {
      return hash('sha384', $this->getSalt() . $this->_password . $this->_getPepper());
    }

    /**
     * Hash the password using an HMAC-inspired hash
     *
     * @access public
     * @return string
     */
    public function getHash()
    {
      return hash('sha512', substr(strrev(self::PEPPER), 0, 20) . $this->_getHash());
    }

  }