﻿#region License

// Copyright (c) 2011 Josue Medrano Cruz
//
// Permission is hereby granted, free of charge, to any person
// obtaining a copy of this software and associated documentation
// files (the "Software"), to deal in the Software without
// restriction, including without limitation the rights to use,
// copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following
// conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
// OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
// HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
// WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.

#endregion


using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Design;
using System.Reflection;
using System.Text.RegularExpressions;
using System.IO;
using System.Linq;
using System.Text;
using System.Web;
using System.Web.UI;
using System.Web.UI.HtmlControls;
using System.Web.UI.WebControls;
using System.ComponentModel;
using System.Web.UI.Design.WebControls;
using System.Data;
using Newtonsoft.Json;
using BorderStyle = System.Web.UI.WebControls.BorderStyle;
using CheckBox = System.Web.UI.WebControls.CheckBox;
using Control = System.Web.UI.Control;
using TextBox = System.Web.UI.WebControls.TextBox;


[assembly: TagPrefix("LookUp", "LookUpSearch")]
[assembly: WebResource("LookUpSearch.LookUp.LookUpScript.js", "text/js")]
[assembly: WebResource("LookUpSearch.Common.LookUp.css", "text/css")]
[assembly: WebResource("LookUpSearch.Common.OuSearch.htm", "text/html")]
[assembly: WebResource("LookUpSearch.Common.Container.htm", "text/html")]
[assembly: WebResource("LookUpSearch.Common.paginate.js", "text/js")]
[assembly: WebResource("LookUpSearch.Common.Close.gif", "image/gif")]
[assembly: WebResource("LookUpSearch.Common.LookUp.gif", "image/gif")]
[assembly: WebResource("LookUpSearch.Common.LookUpDisabled.gif", "image/gif")]
[assembly: WebResource("LookUpSearch.Common.Search.png", "img/png")]
[assembly: WebResource("LookUpSearch.Common.sprite.png", "img/png")]
[assembly: WebResource("LookUpSearch.Common.activity.gif", "image/gif")]
[assembly: WebResource("LookUpSearch.Common.popup_header.png", "img/png")]
[assembly: WebResource("LookUpSearch.Common.Popup_header_table.gif", "img/png")]
[assembly: WebResource("LookUpSearch.Common.lubtn_glass_img.gif", "image/gif")]
[assembly: WebResource("LookUpSearch.Common.lubtn_glass_img_hover.gif", "image/gif")]

internal class LookUpEmbeddedResource
{

}


namespace LookUpSearch
{
    [

    ToolboxData("<{0}:LookUp runat=\"server\"></{0}:LookUp>"),
    Designer(typeof(LookUpDesigner)),
    ToolboxBitmap(typeof(LookUpEmbeddedResource), "LookUpSearch.Common.Search.png")
    ]
    public class LookUp : DataBoundControl, IScriptControl, ICallbackEventHandler, IPostBackEventHandler
    {


        #region private fields


        private bool _doDataBind;
        private int _resultTotalRecords;
        private List<LuTemplate> _cellTemplates;
        private ScriptManager _sm;


        #endregion



        public override void DataBind()
        {
            if (_doDataBind)
                base.DataBind();
        }

        public enum PositionOn
        {
            Control = 1, TargetControl = 2
        }


        public event EventHandler<RowData> OnRowClick;


        #region Design Properties

        [Category("Misc")]
        public TextBox TargetTextBox
        {
            get
            {
                EnsureChildControls();
                TextBox targetTextBoxControl = null;
                if (string.IsNullOrEmpty(TargetControlId))
                {
                    if (!DesignMode)
                    {
                        string msg = string.Format("The TargetControlId property of '{0}' cannot be blank.", ID);
                        throw new Exception(msg);
                    }
                }
                Control nc = NamingContainer;
                try
                {
                    targetTextBoxControl = nc.FindControl(TargetControlId) as TextBox;
                }
                catch (Exception e)
                {
                    if (!DesignMode)
                        throw new Exception(e.Message, e.InnerException);
                }
                return targetTextBoxControl;
            }
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
        [Themeable(false)]
        [PersistenceMode(PersistenceMode.InnerProperty)]
        [TemplateContainer(typeof(LuTemplate))]
        [Browsable(false)]
        public virtual List<LuTemplate> ItemTemplate
        {
            get { return _cellTemplates ?? (_cellTemplates = new List<LuTemplate>()); }
        }

        [Category("Appearance")]
        [Themeable(true)]
        [DefaultValue("")]
        [UrlProperty]
        [Editor("System.Web.UI.Design.UrlEditor, System.Design", typeof(UITypeEditor))]
        [Description("Search template, html part which contains the search controls")]
        public string HtmlTemplate
        {
            get
            {
                var s = ViewState["_HtmlTemplate"];
                return (s == null ? string.Empty : (string)s);
            }

            set
            {
                ViewState["_HtmlTemplate"] = ResolveUrl(value);
            }
        }

        /// <summary>
        /// Gets or sets the image URL.
        /// </summary>
        /// <value>The image URL.</value>
        [Category("Appearance"), DefaultValue("")]
        [Description("The URL of the image to be displayed for the button. The Text property will become the alt text tag for the image button.")]
        [Browsable(true), Editor(typeof(System.Web.UI.Design.UrlEditor), typeof(UITypeEditor))]
        public string ImageUrl
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_ImageUrlOn"];
                return (o == null ? string.Empty : (string)o);
            }

            set
            {
                EnsureChildControls();
                ViewState["_ImageUrlOn"] = ResolveUrl(value);

            }
        }


        [Category("Appearance"), DefaultValue("")]
        [Description("The URL of the image to be displayed for the close button")]
        [Browsable(true), Editor(typeof(System.Web.UI.Design.UrlEditor), typeof(UITypeEditor))]
        public string ImgClose
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_imgClose"];
                return (o == null) ? string.Empty : (string)o;
            }

            set
            {
                EnsureChildControls();
                ViewState["_imgClose"] = ResolveUrl(value);

            }
        }

        [Category("Appearance"), DefaultValue("")]
        [Description("The URL of the image to be displayed for the button. The Text property will become the alt text tag for the image button.")]
        [Browsable(true), Editor(typeof(System.Web.UI.Design.UrlEditor), typeof(UITypeEditor))]
        public string ImageUrlOff
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_ImageUrlOff"];
                return (o == null ? string.Empty : (string)o);
            }

            set
            {
                EnsureChildControls();
                ViewState["_ImageUrlOff"] = ResolveUrl(value);

            }
        }


        [TypeConverterAttribute(typeof(StringArrayConverter)),
        Category("Data"), Description("Fields for which the values will be available when user complete the selection")]
        public string[] DataKeyNames { get; set; }


        /// <summary>
        /// Gets or sets a value indicating whether [dialogue centred].
        /// </summary>
        /// <value><c>true</c> if [dialogue centred]; otherwise, <c>false</c>.</value>
        [Category("Appearance"), DefaultValue(false), Description("Set the dialogue on the center of the window")]
        public bool DialogueCentred
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_DialogueCentred"];
                return (o != null && (bool)o);
            }
            set
            {
                EnsureChildControls();
                ViewState["_DialogueCentred"] = value;
            }
        }



        [Category("Appearance"), DefaultValue(2), Description("Fade delay (seconds)")]
        public int FadeDelay
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_FadeDelay"];
                return (o == null ? 2 : (int)o);
            }
            set
            {
                EnsureChildControls();
                ViewState["_FadeDelay"] = value;
            }
        }


        /// <summary>
        /// Gets or sets the width.
        /// </summary>
        /// <value>The width.</value>
        [Category("Appearance"), DefaultValue(PositionOn.Control), Description("The default for position based on control or Target Control, where the control will appear first where the user invoques the control to pops up")]
        public PositionOn PositionBaseOn
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_PositionBaseOn"];
                return (o == null) ? PositionOn.Control : (PositionOn)o;
            }
            set
            {
                EnsureChildControls();
                ViewState["_PositionBaseOn"] = value;
            }
        }

        /// <summary>
        /// Gets or sets the width.
        /// </summary>
        /// <value>The width.</value>
        [Category("Appearance"), DefaultValue(-1), Description("The default for top position of the windows information")]
        public int PosX
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_PosX"];
                return (o == null ? -1 : (int)o);
            }
            set
            {
                EnsureChildControls();
                ViewState["_PosX"] = value;
            }
        }

        /// <summary>
        /// Gets or sets the width.
        /// </summary>
        /// <value>The width.</value>
        [Category("Appearance"), DefaultValue(-1), Description("The default for bottom position of the windows information")]
        public int PosY
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_PosY"];
                return (o == null ? -1 : (int)o);
            }
            set
            {
                EnsureChildControls();
                ViewState["_PosY"] = value;
            }
        }

        [Category("Appearance"), DefaultValue(10), Description("The number of rows to display in the page.")]
        public int PageSize
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_PageSize"];
                return (o == null ? 10 : (int)o);
            }
            set
            {
                EnsureChildControls();
                ViewState["_PageSize"] = value;
            }
        }

        /// <summary>
        /// Gets or sets the width.
        /// </summary>
        /// <value>The width.</value>
        [Category("Layout"), DefaultValue(400), Description("windows popup Height")]
        public int WHeight
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_wHeight"];
                return (o == null ? 400 : (int)o);
            }
            set
            {
                EnsureChildControls();
                ViewState["_wHeight"] = value;
            }
        }

        /// <summary>
        /// Gets or sets the width.
        /// </summary>
        /// <value>The width.</value>
        [Category("Layout"), DefaultValue(700), Description("windows popup Width")]
        public int WWidth
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_wWidth"];
                return (o == null ? 700 : (int)o);
            }
            set
            {
                EnsureChildControls();
                ViewState["_wWidth"] = value;
            }
        }



        [Category("Behavior"), DefaultValue(false), Description("Indicates wheher the target control is readonly")]
        public bool TargetControlReadOnly
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_TargetControlReadOnly"];
                return (o != null && (bool)o);
            }

            set
            {
                EnsureChildControls();
                ViewState["_TargetControlReadOnly"] = value;
            }
        }


        [Category("Behavior"), DefaultValue(true), Description("Automatically postback after the user has selected a value from result.")]
        public bool AutoPostBack
        {
            get
            {
                var o = ViewState["_AutoPostBack"];
                return (o == null || (bool)o);
            }

            set
            {
                ViewState["_AutoPostBack"] = value;
            }
        }

        [Category("Behavior"), DefaultValue("{0}"), Description("Format how the fields are retrieved based on the DataKeyNames fields, sample: {0} {1} {2}. This is only when no OnRowClick event is handled.")]
        public string TextFormatString
        {
            get
            {
                var o = ViewState["_TextFormatString"];
                return (o == null) ? "{0}" : ((string)o);
            }

            set
            {
                ViewState["_TextFormatString"] = value;
            }
        }



        [Category("Behavior"), DefaultValue(true), Description("Ignores required field attributes in the Template, if all criterias are empty, search will continue. Attribute in template isrequired=\"true\" ")]
        public bool IgnoreRequiredAttributes
        {
            get
            {
                var o = ViewState["_IgnoreRequiredAttributes"];
                return ((o == null) || (bool)o);
            }

            set
            {
                ViewState["_IgnoreRequiredAttributes"] = value;
            }
        }


        [Category("Behavior")]
        [DefaultValue(200)]
        [Description("Defines the maximum entries displayed in the lookup table.")]
        public int MaximumEntries
        {
            get
            {
                var o = ViewState["MaximumEntries"];
                return (o == null) ? 200 : (int)o;
            }

            set
            {
                ViewState["MaximumEntries"] = value;
            }
        }

        /// <summary>
        /// Contol which event will cause the popup to show
        /// </summary>
        /// <value>The target control id.</value>
        [Category("Misc")]
        [DefaultValue(""), Bindable(false), Description("Indicate the TextBox Control which is being extended or targered for the look up.")]
        [Editor(typeof(AllControlsEditor), typeof(UITypeEditor)), Browsable(true)]
        public string TargetControlId
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_TargetControlID"];
                return (o == null ? string.Empty : (string)o);
            }
            set
            {
                EnsureChildControls();
                ViewState["_TargetControlID"] = value;
            }
        }

        [Category("Behavior")]
        [DefaultValue(true), Bindable(false), Description("Control to assign value.")]
        public bool ExtenderEnabled
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_ExtenderEnabled"];
                return (o == null || (bool)o);
            }

            set
            {
                EnsureChildControls();
                ViewState["_ExtenderEnabled"] = value;
            }
        }


        [Category("Behavior")]
        [DefaultValue(true), Bindable(false), Description("Indicates whether user can select multible rows from the records result.")]
        public bool MultiSelection
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_MultiSelection"];
                return (o == null || (bool)o);
            }

            set
            {
                EnsureChildControls();
                ViewState["_MultiSelection"] = value;
            }
        }

        [Category("Accessibility")]
        [DefaultValue(""), Bindable(false), Description("Title for the search dialogue")]
        public string TitlePopUp
        {
            get
            {
                EnsureChildControls();
                var o = ViewState["_TitlePopUp"];
                return (o == null ? string.Empty : (string)o);
            }

            set
            {
                EnsureChildControls();
                ViewState["_TitlePopUp"] = value;
            }
        }


        #region private properties or not visible for designer


        [Category("Data")]
        private ObjectDataSource ObjObjectDataSource
        {
            get
            {
                EnsureChildControls();
                ObjectDataSource objObjectDataSource = null;
                if (string.IsNullOrEmpty(DataSourceID))
                {
                    if (!DesignMode)
                    {
                        string msg = string.Format("The TargetControlId property of '{0}' cannot be blank.", ID);
                        throw new Exception(msg);
                    }
                }
                Control nc = NamingContainer;
                try
                {
                    objObjectDataSource = nc.FindControl(DataSourceID) as ObjectDataSource;
                }
                catch (Exception e)
                {
                    if (!DesignMode)
                    {
                        throw new Exception(e.Message, e.InnerException);
                    }
                }

                return objObjectDataSource;

            }
        }

        private string ColumnHeader
        {
            get
            {
                EnsureChildControls();

                string columTitle = string.Join("°°", ItemTemplate.Select(column => column.HeaderText).ToArray());
                return columTitle;
            }

        }

        [Browsable(false)]
        public override bool EnableTheming
        {
            get;
            set;
        }

        [Browsable(false)]
        public override Color BackColor
        {
            get;
            set;
        }
        [Browsable(false)]
        public override Color BorderColor
        {
            get;
            set;
        }
        [Browsable(false)]
        public override BorderStyle BorderStyle
        {
            get;
            set;
        }
        [Browsable(false)]
        public override Unit BorderWidth
        {
            get;
            set;
        }
        [Browsable(false)]
        public override string CssClass
        {
            get;
            set;
        }
        [Browsable(false)]
        public override FontInfo Font
        {
            get
            {
                return base.Font;
            }
        }
        [Browsable(false)]
        public override Color ForeColor
        {
            get;
            set;
        }

        #endregion

        #endregion

        #region Implementation of IScriptControl


        /// <summary>
        /// Gets a collection of <see cref="T:System.Web.UI.ScriptReference"/> objects that define script resources that the control requires.
        /// </summary>
        /// <returns>
        /// An <see cref="T:System.Collections.IEnumerable"/> collection of <see cref="T:System.Web.UI.ScriptReference"/> objects.
        /// </returns>
        public IEnumerable<ScriptReference> GetScriptReferences()
        {

            //if (!Page.Items.Contains("ScriptsAlreadyPut"))
            //{
            //    Page.Items.Add("ScriptsAlreadyPut", "ScriptsAlreadyPut");
            //}
            var refHtmlLookUp = new ScriptReference("LookUpSearch.LookUp.LookUpScript.js", GetType().Assembly.FullName);
            var refPaginate = new ScriptReference("LookUpSearch.Common.paginate.js", GetType().Assembly.FullName);

            return new[] { refHtmlLookUp, refPaginate };

        }

        /// <summary>
        /// Gets a collection of script descriptors that represent ECMAScript (JavaScript) client components.
        /// </summary>
        /// <returns>
        /// An <see cref="T:System.Collections.IEnumerable"/> collection of <see cref="T:System.Web.UI.ScriptDescriptor"/> objects.
        /// </returns>
        public IEnumerable<ScriptDescriptor> GetScriptDescriptors()
        {

            Page.ClientScript.GetCallbackEventReference(this, "data", string.Format("{0}_ReceiveData", UniqueID), "context");

            var scDesc = new ScriptControlDescriptor("LookUpSearch.LookUpControl", ClientID);

            if (!DesignMode)
            {

                scDesc.AddProperty("textBoxId", TargetTextBox.ClientID);
                scDesc.AddProperty("uniqueId", UniqueID);
                scDesc.AddProperty("clientId", ClientID);
                scDesc.AddProperty("wWidth", WWidth);
                scDesc.AddProperty("wHeight", WHeight);
                scDesc.AddProperty("posX", PosX);
                scDesc.AddProperty("posY", PosY);
                scDesc.AddProperty("pageSize", PageSize);
                scDesc.AddProperty("iFrameUrl", "javascript:false;");
                scDesc.AddProperty("dialogueImgClose", string.IsNullOrEmpty(ImgClose) ? Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.Close.gif") : ImgClose);
                scDesc.AddProperty("enabled", (Enabled && TargetTextBox.Enabled));
                scDesc.AddProperty("PositionBaseOn", PositionBaseOn);
                scDesc.AddProperty("DialogueCentred", DialogueCentred);
                scDesc.AddProperty("multiSelection", MultiSelection);
                scDesc.AddProperty("titlePopUp", TitlePopUp);
                scDesc.AddProperty("columnHeaders", ColumnHeader);
                scDesc.AddProperty("spriteImg", Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.sprite.png"));
                scDesc.AddProperty("progressImageUrl", Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.activity.gif"));
                scDesc.AddProperty("popupHeaderUrl", Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.popup_header.png"));
                scDesc.AddProperty("popupHeaderTableUrl", Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.Popup_header_table.gif"));
                scDesc.AddProperty("HtmTemplate", GetHtmlTemplate());
                scDesc.AddProperty("HtmContainer", GetHtmlContainer());
                scDesc.AddProperty("SelectParameters", GetParamList());
                scDesc.AddProperty("maxRecordsToShow", MaximumEntries);
                scDesc.AddProperty("ignoreRequiredAttributes", IgnoreRequiredAttributes);
                scDesc.AddProperty("fadeDelay", FadeDelay);
                scDesc.AddProperty("lubtn_glass_img", Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.lubtn_glass_img.gif"));
                scDesc.AddProperty("lubtn_glass_img_hover", Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.lubtn_glass_img_hover.gif"));



            }
            return new[] { scDesc };
        }

        private string GetHtmlTemplate()
        {
            var htmlString = string.Empty;

            //If not template, take default
            if (string.IsNullOrEmpty(HtmlTemplate))
            {
                var assembly = Assembly.GetExecutingAssembly();
                Stream str = assembly.GetManifestResourceStream("LookUpSearch.Common.OuSearch.htm");
                if (str != null)
                {
                    var rs = new StreamReader(str, Encoding.ASCII);
                    htmlString = rs.ReadToEnd();
                }
            }
            else
            {
                htmlString = File.ReadAllText(HttpRuntime.AppDomainAppPath + HtmlTemplate);
            }
            return htmlString;

        }

        private string GetHtmlContainer()
        {
            var htmlString = string.Empty;
            var assembly = Assembly.GetExecutingAssembly();
            var str = assembly.GetManifestResourceStream("LookUpSearch.Common.Container.htm");
            if (str != null)
            {
                var rs = new StreamReader(str, Encoding.ASCII);
                htmlString = rs.ReadToEnd();
                return htmlString;
            }
            return htmlString;
        }


        private string GetParamList()
        {
            var strParam = string.Empty;
            foreach (Parameter para in ObjObjectDataSource.SelectParameters)
            {
                if (strParam.Length > 0)
                    strParam += ";";
                strParam += para.Name;
            }
            return strParam;

        }
        #endregion

        /// <summary>
        /// Raises the <see cref="E:System.Web.UI.Control.PreRender"/> event.
        /// </summary>
        /// <param name="e">An <see cref="T:System.EventArgs"/> object that contains the event data.</param>
        protected override void OnPreRender(EventArgs e)
        {

            if (!DesignMode)
            {
                // Test for ScriptManager and register if it exists

                _sm = ScriptManager.GetCurrent(Page);
                if (_sm == null)
                    throw new HttpException("A ScriptManager control must exist on the current page.");

                _sm.RegisterScriptControl(this);
                if (TargetControlReadOnly)
                {
                    TargetTextBox.ReadOnly = false;
                    TargetTextBox.Attributes.Add("readonly", "readonly");
                }

                if (!Page.Items.Contains("styleAlreadyPut"))
                {
                    var csslink = new HtmlGenericControl("link");
                    csslink.Attributes.Add("href", Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.LookUp.css"));
                    csslink.Attributes.Add("type", "text/css");
                    csslink.Attributes.Add("rel", "stylesheet");
                    Page.Header.Controls.Add(csslink);

                    var ltr = new LiteralControl();
                    const string cssclass = @"<style type=""text/css"" rel=""stylesheet"">
                            .lubButton_Glass {{
                                color: #000000;
                                line-height: normal;
                                padding: 1px;
                                font-family: Tahoma, Geneva, sans-serif;
                                font-size: 12px;
                                font-style: normal;
                                font-variant: normal;
                                font-weight: normal;
                                vertical-align: middle;
                                border-color: #b2bcbe;
                                border-width: 1px;
                                border-style: solid;
                                cursor: pointer;
                                font-size-adjust: none;
                                font-stretch: normal;
                                background-attachment: scroll;
                                background-repeat: repeat-x;
                                background-position-x: center;
                                background-position-y: top;
                                background-size: auto;
                                background-origin: padding-box;
                                background-clip: border-box;
                                background-color: rgb(218, 223, 224);
                                background-image: url(""{0}"");
                            }}

                                .lubButton_Glass:hover {{
                                    border-color: #71acb4;
                                    border-width: 1px;
                                    border-style: solid;
                                    background-repeat: repeat-x;
                                    background-position-x: center;
                                    background-position-y: top;
                                    background-size: auto;
                                    background-origin: padding-box;
                                    background-clip: border-box;
                                    background-color: rgb(181, 215, 219);
                                    background-image: url(""{1}"");
                                }}

                                .lubButton_Glass div.lubox {{
                                    padding-top: 3px;
                                    padding-right: 13px;
                                    padding-bottom: 4px;
                                    padding-left: 13px;
                                    border-top-width: 0px;
                                    border-right-width: 0px;
                                    border-bottom-width: 0px;
                                    border-left-width: 0px;
                                }}

                    </style>";

                    ltr.Text = string.Format(cssclass,
                        Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.lubtn_glass_img.gif"), Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.lubtn_glass_img_hover.gif"));
                    Page.Header.Controls.Add(ltr);
                    
                    
                    
                    Page.Items.Add("styleAlreadyPut", "styleAlreadyPut");


                }

            }

            base.OnPreRender(e);

        }

        /// <summary>
        /// Renders the control to the specified HTML writer.
        /// </summary>
        /// <param name="writer">The <see cref="T:System.Web.UI.HtmlTextWriter"/> object that receives the control content.</param>

        protected override void Render(HtmlTextWriter writer)
        {


            if (!DesignMode)
                _sm.RegisterScriptDescriptors(this);



            writer.AddAttribute(HtmlTextWriterAttribute.Cellpadding, "1", false);
            writer.AddAttribute(HtmlTextWriterAttribute.Border, "0");
            writer.AddAttribute(HtmlTextWriterAttribute.Width, Width.ToString());
            writer.AddAttribute(HtmlTextWriterAttribute.Height, Height.ToString());
            writer.AddAttribute(HtmlTextWriterAttribute.Style, "border-collapse:collapse; display:inline;");

            writer.RenderBeginTag(HtmlTextWriterTag.Table);
            writer.RenderBeginTag(HtmlTextWriterTag.Tr);
            writer.RenderBeginTag(HtmlTextWriterTag.Td);
            writer.WriteBeginTag("img");
            writer.WriteAttribute("Id", ClientID);
            writer.WriteAttribute("alt", ToolTip);
            writer.WriteAttribute("title", ToolTip);

            string imageUrl;

            var pImageUrl = ImageUrl;
            var pImageUrlOff = ImageUrlOff;


            if (string.IsNullOrEmpty(pImageUrl))
                pImageUrl = Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.LookUp.gif");

            if (string.IsNullOrEmpty(pImageUrlOff))
                pImageUrlOff = Page.ClientScript.GetWebResourceUrl(GetType(), "LookUpSearch.Common.LookUpDisabled.gif");

            if (!DesignMode)
            {
                if (Enabled && TargetTextBox.Enabled)
                    writer.WriteAttribute("style", "cursor:pointer;");


                if (Enabled && TargetTextBox.Enabled)
                    imageUrl = pImageUrl;
                else
                    imageUrl = pImageUrlOff;
            }
            else
                imageUrl = pImageUrl;

            if (DesignMode)
                imageUrl = imageUrl.Replace("~/", "");


            writer.WriteAttribute("src", imageUrl);
            writer.Write(HtmlTextWriter.TagRightChar);

            //writer.AddAttribute("onmouseover", "this.src='" + base.ResolveClientUrl(imageOverUrl) + "'");
            //writer.AddAttribute("onmouseout", "this.src='" + base.ResolveClientUrl(imageUrl) + "'");

            writer.WriteEndTag("img");
            writer.WriteLine();

            writer.AddStyleAttribute("align", "absmiddle");

            writer.Write(" ");
            writer.RenderEndTag();
            writer.RenderEndTag();
            writer.RenderEndTag();
            writer.WriteLine("");


        }


        #region Implementation of ICallbackEventHandler

        /// <summary>
        /// Processes a callback event that targets a control.
        /// </summary>
        /// <param name="eventArgument">A string that represents an event argument to pass to the event handler.</param>
        public void RaiseCallbackEvent(string eventArgument)
        {

            //string json = JsonConvert.SerializeObject(ObjObjectDataSource.SelectParameters);

            ObjObjectDataSource.SelectParameters.Clear();

            var jarray = JsonConvert.DeserializeObject<List<SelectParameter>>(eventArgument);

            foreach (var selectParameter in jarray)
            {
                ObjObjectDataSource.SelectParameters.Add(selectParameter.ParamName, selectParameter.Value);
            }

        }

        /// <summary>
        /// Returns the results of a callback event that targets a control.
        /// </summary>
        /// <returns>The result of the callback.</returns>
        public string GetCallbackResult()
        {
            return RenderGridList();
        }

        #region RenderGridList

        protected virtual string RenderGridList()
        {
            var sb = new StringBuilder(1024);

            _doDataBind = true;
            DataSourceView dsv = GetData();
            //this.DataBind();

            try
            {

                dsv.Select(new DataSourceSelectArguments(), delegate(IEnumerable data)
                {
                    if (data == null)
                        sb.Append(string.Empty);
                    else
                    {
                        var table = new Table();
                        var rowId = 0;
                        foreach (object o in data)
                        {

                            var row = new TableRow();

                            var lvalueInKeys = new List<string>();
                            foreach (var item in DataKeyNames)
                            {
                                var valueInKey = DataBinder.Eval(o, item);
                                lvalueInKeys.Add(valueInKey.ToString());
                            }

                            string valueInKeys = JsonConvert.SerializeObject(lvalueInKeys);

                            if (MultiSelection)
                            {
                                var tc = new TableCell {CssClass = "LU_CellLeft"};
                                var ctrl = new CheckBox { ID = String.Format("RId{0}_{1}", rowId, ClientID) };
                                ctrl.InputAttributes.Add("value", valueInKeys);
                                ctrl.InputAttributes.Add("ismultiCb", "true");
                                tc.Controls.Add(ctrl);
                                row.Cells.Add(tc);
                            }

                            BindRow(o, ref row);

                            if (!MultiSelection)
                                row.Attributes.Add("onclick", Page.ClientScript.GetPostBackEventReference(this, string.Format("{0}", valueInKeys)));

                            table.Rows.Add(row);

                            rowId++;

                            if (rowId == MaximumEntries)
                                break;

                        }

                        var newList = (from e in data.Cast<object>() select e).ToList();
                        _resultTotalRecords = newList.Count();

                        using (var sw = new StringWriter(sb))
                        {
                            using (var writer = new HtmlTextWriter(sw))
                            {
                                table.RenderControl(writer);
                            }
                        }

                    }

                });
            }
            catch (Exception ex)
            {

                var table = new Table();
                var infoRow = CreateInfoRow(ex.Message);

                infoRow.Style.Add("background-color", "#FFCECE");
                table.Rows.Add(infoRow);
                _resultTotalRecords = 1;

                using (var sw = new StringWriter(sb))
                {
                    using (var writer = new HtmlTextWriter(sw))
                    {
                        table.RenderControl(writer);
                    }
                }



            }
            _doDataBind = false;

            var result = sb.ToString();
            result = Regex.Replace(result, @"<.*?table(.*?)>", "", RegexOptions.Compiled | RegexOptions.Multiline);
            return JsonConvert.SerializeObject(new JsonResult { ResultTotalRecords = _resultTotalRecords, HtmlData = result });

        }

        private void BindRow(object data, ref TableRow row)
        {
            //TableRow row = new TableRow();
            row.Attributes["rowType"] = "data";

            if (_cellTemplates != null && _cellTemplates.Count != 0)
            {
                //List<Control> controls = new List<Control>();
                int columnIndex = -1;
                foreach (var template in this._cellTemplates)
                {
                    columnIndex++;
                    template.DataBind();
                    var item = new TableCellItem(data, 0);
                    template.Template.InstantiateIn(item);
                    item.DataBind();
                    item.CssClass = template.CssClass;
                    row.Cells.Add(item);
                }
            }

        }


        private TableRow CreateInfoRow(string msg)
        {
            var row = new TableRow();
            var cell = new TableCell();

            row.Attributes["rowType"] = "info"; // info row 

            if (_cellTemplates == null)
            {
                _cellTemplates = new List<LuTemplate>();
                var msgrow = new LuTemplate();
                _cellTemplates.Add(msgrow);

            }

            cell.ColumnSpan = this._cellTemplates.Count + (MultiSelection ? 1 : 0);
            row.Cells.Add(cell);


            cell.Text = msg;
            cell.HorizontalAlign = HorizontalAlign.Left;

            return row;
        }

        #endregion

        #endregion

        #region Implementation of IPostBackEventHandler

        /// <summary>
        /// Enables a server control to process an event raised when a form is posted to the server.
        /// </summary>
        /// <param name="eventArgument">A <see cref="T:System.String"/> that represents an optional event argument to be passed to the event handler.</param>
        public void RaisePostBackEvent(string eventArgument)
        {
            var dataRow = new RowData(eventArgument, MultiSelection, DataKeyNames);

            if (OnRowClick != null)
            {
                // var dataResult = new RowEventArgs(DataRow);
                OnRowClick(this, dataRow);
            }
            else
            {
                if (dataRow.DataTable.Rows.Count > 0)
                {
                    if (!string.IsNullOrEmpty(TextFormatString))
                    {
                        try
                        {
                            DataRow dr = dataRow.DataTable.Rows[0];
                            string[] resultArray = dr.ItemArray.Select(dc => dc.ToString()).ToArray();
                            TargetTextBox.Text = string.Format(TextFormatString, resultArray);
                        }
                        catch { }
                    }


                }
            }
        }


        #endregion

        # region complement class and interfaces

        private class JsonResult
        {
            public int ResultTotalRecords { get; set; }
            public string HtmlData { get; set; }
        }


        internal class SelectParameter 
        {
            public string ParamName { get; set; }
            public string Value { get; set; }

        }


        /// <summary>
        /// Row data returned
        /// </summary>
        public class RowData : EventArgs
        {
            public DataTable DataTable = new DataTable();

            public RowData(string eventArgument, bool multiselection, string[] DataKeyNames)
            {
                foreach (string column in DataKeyNames)
                {
                    DataTable.Columns.Add(column, typeof(string));
                }
                if (string.IsNullOrEmpty(eventArgument))
                    return;

                string[] rows = Regex.Split(eventArgument, @"\s*°°\s*");
                foreach (var row in rows)
                {
                    var cols = JsonConvert.DeserializeObject<string[]>(row);
                    DataTable.Rows.Add(cols);
                }

            }
        }

        #endregion

        #region Implementation of Control Type


        /// <summary>
        /// Editor for selecting all Asp.Net controls
        /// </summary>
        public class AllControlsEditor : ControlsEditor
        {
            #region Members

            /// <summary>
            /// Invoke base constructor
            /// </summary>
            public AllControlsEditor() : base(typeof(TextBox)) { }

            #endregion
        }


        /// <summary>
        /// Editor for selecting controls from Asp.Net page
        /// </summary>
        public abstract class ControlsEditor : UITypeEditor
        {
            #region Variables

            private System.Windows.Forms.Design.IWindowsFormsEditorService edSvc = null;
            private System.Windows.Forms.ListBox lb;
            private readonly Type typeShow;
            int num;
            #endregion

            #region Constructor

            /// <summary>
            /// onstructor - show specified types
            /// </summary>
            /// <param name="show">Type descriptor</param>
            protected ControlsEditor(Type show)
            {
                typeShow = show;
            }

            #endregion

            #region Methods


            /// <summary>
            /// Edits the specified object's value using the editor style indicated by the <see cref="M:System.Drawing.Design.UITypeEditor.GetEditStyle"/> method.
            /// </summary>
            /// <param name="context">An <see cref="T:System.ComponentModel.ITypeDescriptorContext"/> that can be used to gain additional context information.</param>
            /// <param name="provider">An <see cref="T:System.IServiceProvider"/> that this editor can use to obtain services.</param>
            /// <param name="value">The object to edit.</param>
            /// <returns>
            /// The new value of the object. If the value of the object has not changed, this should return the same object it was passed.
            /// </returns>
            public override object EditValue(ITypeDescriptorContext context,
              IServiceProvider provider, object value)
            {

                if (context.Instance != null)
                {
                    edSvc = (System.Windows.Forms.Design.IWindowsFormsEditorService)
                      provider.GetService(typeof(System.Windows.Forms.Design.IWindowsFormsEditorService));
                    if (edSvc != null)
                    {
                        lb = new System.Windows.Forms.ListBox();
                        lb.BorderStyle = System.Windows.Forms.BorderStyle.None;
                        lb.SelectedIndexChanged += lb_SelectedIndexChanged;

                        ArrayList separator = new ArrayList { "" };

                        int num2 = context.Container.Components.Count - 1;
                        for (num = 0; num <= num2; num++)
                        {
                            if (context.Container.Components[num].Site.Component.GetType() == typeShow)
                            {
                                separator.Add(context.Container.Components[num].Site.Name.Trim());
                                lb.Items.Add(context.Container.Components[num].Site.Name.Trim());
                            }
                        }

                        separator.Sort();

                        if (lb.Items.Count > 0)
                            edSvc.DropDownControl(lb);

                        if (lb.SelectedIndex == -1)
                            return value;
                        return lb.SelectedItem;
                    }
                }

                return value;
            }



            /// <summary>
            /// Gets the editor style used by the <see cref="M:System.Drawing.Design.UITypeEditor.EditValue(System.IServiceProvider,System.Object)"/> method.
            /// </summary>
            /// <param name="context">An <see cref="T:System.ComponentModel.ITypeDescriptorContext"/> that can be used to gain additional context information.</param>
            /// <returns>
            /// A <see cref="T:System.Drawing.Design.UITypeEditorEditStyle"/> value that indicates the style of editor used by the <see cref="M:System.Drawing.Design.UITypeEditor.EditValue(System.IServiceProvider,System.Object)"/> method. If the <see cref="T:System.Drawing.Design.UITypeEditor"/> does not support this method, then <see cref="M:System.Drawing.Design.UITypeEditor.GetEditStyle"/> will return <see cref="F:System.Drawing.Design.UITypeEditorEditStyle.None"/>.
            /// </returns>
            public override UITypeEditorEditStyle GetEditStyle(ITypeDescriptorContext context)
            {
                return UITypeEditorEditStyle.DropDown;
            }

            /// <summary>
            /// Handles the SelectedIndexChanged event of the lb control.
            /// </summary>
            /// <param name="sender">The source of the event.</param>
            /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
            private void lb_SelectedIndexChanged(object sender, EventArgs e)
            {
                if (edSvc != null)
                {
                    edSvc.CloseDropDown();
                }
            }

            #endregion
        }
        #endregion

        #region Implementation of Control Designer
        /// <summary>
        /// LookUpDesigner
        /// </summary>
        private class LookUpDesigner : DataBoundControlDesigner
        {
            protected override string GetErrorDesignTimeHtml(Exception ex)
            {
                return CreateErrorDesignTimeHtml(ex.Message);
            }

            protected override bool ConnectToDataSource()
            {
                return true;
            }

            /// <summary>
            /// Returns a design view of the control as rendered by the control itself.
            /// </summary>
            /// <returns>The HTML of the design time control.</returns>
            public override string GetDesignTimeHtml()
            {

                var cb = (LookUp)Component;
                var sw = new StringWriter();
                var tw = new HtmlTextWriter(sw);

                cb.RenderBeginTag(tw);
                cb.RenderControl(tw);
                cb.RenderEndTag(tw);
                return (sw.ToString());

            }

        }
        #endregion
    }

}
