﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security;
using System.Security.Permissions;
using System.Web;
using System.Web.Mvc;
using System.Text;
using Resources;
//
using MvcBasic.Logic;
using MvcBasicSite.Models.Grid;
using MvcBasicSite.Models.Helpers;

namespace MvcBasicSite.Controllers
{
    /// <summary>
    /// Defines the site document controller.
    /// </summary>
    [ValidateInput(false)]
    public class SiteDocumentController : BaseController
    {
        /// <summary>
        /// The flag used to notify that the view was created from Index method.
        /// </summary>
        private static bool _fromIndex = false;

        /// <summary>
        /// Show the view with all site documents.
        /// </summary>
        /// <returns>The index view.</returns>
        [Authorize]
        public ActionResult Index()
        {
            _fromIndex = true;
            //
            return View();
        }

        /// <summary>
        /// Edit the file (HTML or TXT) for a site document.
        /// </summary>
        /// <param name="id">The site document ID.</param>
        /// <returns>The view for editing the file.</returns>
        [Authorize]
        public ActionResult Edit(int id)
        {
            SiteDocument SiteDocument = _db.SiteDocuments.Single(u => u.ID == id);
            try
            {
                string filePath = Path.Combine(HttpContext.Server.MapPath("/Content/Doc"), Path.GetFileName(SiteDocument.FileFullName));
                Session["FilePath"] = filePath;
                //
                using (var stream = new StreamReader(filePath))
                {
                    SiteDocument.FileData = stream.ReadToEnd();
                }
            }
            catch (Exception ex)
            {
                MvcBasicLog.LogException(ex);
                ModelState.AddModelError("", ex.Message);
                //
                return View(SiteDocument);
            }
            //
            return View(SiteDocument);
        }

        /// <summary>
        /// Postback action for saving the current site document.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns>The result view.</returns>
        [HttpPost]
        public ActionResult Edit(SiteDocument model)
        {
            string filePath = (string)Session["FilePath"];
            //
            try
            {
                FileInfo fileInfo = new System.IO.FileInfo(filePath);
                try
                {
                    //
                    // Used only for local testing!
                    //
                    fileInfo.IsReadOnly = false;
                }
                catch
                {
                    //Ignore it!
                }
                //
                using (FileStream fileStream = new FileStream(filePath, FileMode.Truncate))
                {
                    System.IO.StreamWriter file = new System.IO.StreamWriter(fileStream);
                    file.WriteLine(model.FileData);
                    file.Close();
                }
            }
            catch (Exception ex)
            {
                MvcBasicLog.LogException(ex);
                ModelState.AddModelError("", ex.Message);
                //
                return View(model);
            }
            //
            return RedirectToAction("Index");
        }


        /// <summary>
        /// Delete the selected SiteDocument.
        /// </summary>
        /// <param name="id">The ID of the documents.</param>
        /// <param name="gridSettings">The grid settings.</param>
        /// <returns>Users view.</returns>
        [Authorize]
        public ActionResult Delete(int id, string gridSettings)
        {
            if (gridSettings != null)
                Session["SiteDocumentGridSettings"] = gridSettings; // Cache the Grid Settings!
            //
            SiteDocument SiteDocument = _db.SiteDocuments.Single(u => u.ID == id);
            return View(SiteDocument);
        }

        /// <summary>
        /// Postback confirmation for deleting a site document.
        /// </summary>
        /// <param name="id">The site document ID.</param>
        /// <returns>Redirection to the index of the page.</returns>
        [HttpPost, ActionName("Delete")]
        public ActionResult DeleteConfirmed(int id)
        {
            SiteDocument SiteDocument = _db.SiteDocuments.Single(u => u.ID == id);
            //
            try
            {
                _db.SiteDocuments.DeleteObject(SiteDocument);
                _db.SaveChanges();
            }
            catch (Exception ex)
            {
                MvcBasicLog.LogException(ex);
                ModelState.AddModelError("", Resource.SiteDocumentDeleteError);
                return View("Delete", SiteDocument);
            }
            //
            return RedirectToAction("Index");
        }

        /// <summary>
        /// Load the data that will be used by the jqGrid by using the given grid settings.
        /// </summary>
        /// <param name="grid">The grid sesstings.</param>
        /// <returns>The data in JSON format.</returns>
        public JsonResult GetData(GridSettings grid)
        {
            if (_fromIndex && Session["SiteDocumentGridSettings"] != null)
            {
                //
                // Get grid settings from cache!
                //
                grid = new GridSettings((string)Session["SiteDocumentGridSettings"]);
            }
            //
            _fromIndex = false; // Must be set on false here!
            //
            // Load the data from the database for the current grid settings.
            //
            int count;
            var query = grid.LoadGridData<SiteDocument>(SiteDocument.GetAllSiteDocuments(_db), out count);
            var data = query.ToArray();
            //
            // Convert the results in grid format.
            //
            string gridSettingsString = grid.ToString(); // Used to preserve grid settings!
            Session["SiteDocumentGridSettings"] = gridSettingsString;
            gridSettingsString = null;
            var result = new
            {
                total = (int)Math.Ceiling((double)count / grid.PageSize),
                page = grid.PageIndex,
                records = count,
                rows = (from host in data
                        select new
                        {
                            ID = host.ID,
                            FileFullName = host.FileFullName,
                            Name = host.Name,
                            Culture = RenderHelpers.EnumDropDownList(this, "dropDown", "SiteDocument/SetCultureInfo",
                                host.Culture != null ? (SiteCultures)host.Culture : SiteCultures.All, host.IsSystemDoc == null ? false : true),
                            IsNotPublic = RenderHelpers.CustomCheckBox(this, "checkBox1", "SiteDocument/SetIsNotPublic",
                                host.IsNotPublic != null && host.IsNotPublic == true ? "1" : "0", host.IsSystemDoc == null ? false : true),
                            IsForAgreement = RenderHelpers.CustomCheckBox(this, "checkBox2", "SiteDocument/SetIsForAgreement",
                                host.IsForAgreement != null && host.IsForAgreement == true ? "1" : "0", host.IsSystemDoc == null ? false : true),
                            Date = host.Date.ToString(),
                            Actions = string.Format("{0} {1}",
                                host.IsEditable ?
                                   RenderHelpers.ImageButton(this, Resource.EditTip, "~/Content/Images/Edit.gif", "Edit", new { id = host.ID }, new { style = "border:0px;" })
                                   : RenderHelpers.ImageButton(this, Resource.ViewTip, "~/Content/Images/view.png", "GetFileResult", new { id = host.ID }, new { style = "border:0px;" }, new { target = "blank_" }),
                                host.IsSystemDoc == null || host.IsSystemDoc == false ?
                                   RenderHelpers.ImageButton(this, Resource.DeleteTip, "~/Content/Images/Delete.gif", "Delete", new { id = host.ID, gridSettings = gridSettingsString }, new { style = "border:0px;" })
                                   : string.Empty)
                        }).ToArray()
            };
            //
            // Convert to JSON before to return.
            //
            return Json(result, JsonRequestBehavior.AllowGet);
        }



        /// <summary>
        /// Upload files to the server.
        /// </summary>
        /// <param name="uploadFile">The uploaded file metadata.</param>
        /// <returns>The action result.</returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult FileUpload(HttpPostedFileBase uploadFile)
        {
            if (uploadFile != null)
            {
                if (uploadFile.ContentLength > 0)
                {
                    if (uploadFile.FileName.ToLower().EndsWith(".htm") || uploadFile.FileName.ToLower().EndsWith(".html")
                        || uploadFile.FileName.ToLower().EndsWith(".txt") || uploadFile.FileName.ToLower().EndsWith(".pdf")
                        || uploadFile.FileName.ToLower().EndsWith(".jpg") || uploadFile.FileName.ToLower().EndsWith(".png"))
                    {
                        string filePath = Path.Combine(HttpContext.Server.MapPath("../Content/Doc"), Path.GetFileName(uploadFile.FileName));
                        uploadFile.SaveAs(filePath);
                        int i = uploadFile.FileName.LastIndexOf("\\");
                        string fileName = uploadFile.FileName.Substring(i + 1);
                        //
                        //Create a new SiteDocument and save it into the database!
                        //
                        SiteDocument doc = new SiteDocument
                        {
                            Name = PrepareDocumentName(fileName),
                            FileFullName = (Request.IsLocal ? filePath : uploadFile.FileName),
                            SiteDocumentCulture = SiteCultures.All,
                            IsNotPublic = null,
                            Date = DateTime.Now,
                            IsSystemDoc = null
                        };
                        //
                        try
                        {
                            _db.SiteDocuments.AddObject(doc);
                            _db.SaveChanges();
                        }
                        catch (Exception ex)
                        {
                            MvcBasicLog.LogException(ex);
                            ViewData.ModelState.AddModelError("", Resource.SaveInDBError);
                        }
                    }
                    else
                    {
                        ViewData.ModelState.AddModelError("", string.Format("{0} {1} {2}",
                            Resource.FileUploaderYouSelected, uploadFile.ContentType, Resource.FileUploaderFile));
                    }
                }
            }
            //
            return View("Index");
        }

        /// <summary>
        /// Remove the extension of the file for the UI customisation.
        /// </summary>
        /// <param name="uploadFileName">The file name.</param>
        /// <returns>The document name ready for UI.</returns>
        private string PrepareDocumentName(string uploadFileName)
        {
            string temp = uploadFileName.Substring(0, uploadFileName.LastIndexOf('.'));
            //
            return (temp.Length > 50 ? temp.Substring(0, 50) : temp);
        }

        /// <summary>
        /// Set the culture info.
        /// </summary>
        /// <param name="value">The culture value.</param>
        /// <param name="id">The row id.</param>
        /// <returns>The action result.</returns>
        [Authorize]
        public ActionResult SetCultureInfo(string value, string id)
        {
            int _id = 0;
            bool tryParseRowID = Int32.TryParse(id, out _id);
            //
            if (tryParseRowID)
            {
                SiteCultures culture = (SiteCultures)Enum.Parse(typeof(SiteCultures), value, true);
                {
                    SiteDocument SiteDocument = _db.SiteDocuments.Single(u => u.ID == _id);
                    if (SiteDocument != null)
                    {
                        SiteDocument.Culture = (int?)culture;
                        try
                        {
                            _db.SaveChanges();
                        }
                        catch (Exception ex)
                        {
                            MvcBasicLog.LogException(ex);
                            ModelState.AddModelError("", Resource.SaveInDBError);
                        }
                    }
                }
            }
            //
            return View("Index");
        }

        /// <summary>
        /// Save the changed checkbox value from UI corresponding to the SetIsNotPublic property.
        /// </summary>
        /// <param name="rowid">The rowID.</param>
        /// <param name="value">The checkbox value.</param>
        /// <returns>The axtion result.</returns>
        [Authorize]
        public ActionResult SetIsNotPublic(int rowid, string value)
        {
            bool oldValueCheked = value == "1";
            SiteDocument SiteDocument = _db.SiteDocuments.Single(u => u.ID == rowid);
            //
            if (SiteDocument != null)
            {
                SiteDocument.IsNotPublic = (oldValueCheked || SiteDocument.IsForAgreement == true) ? (bool?)null : true;
                //
                try
                {
                    _db.SaveChanges();
                }
                catch (Exception ex)
                {
                    MvcBasicLog.LogException(ex);
                    ModelState.AddModelError("", Resource.SaveInDBError);
                }
            }
            //
            return View("Index");
        }

        /// <summary>
        /// Save the changed checkbox value from UI corresponding to the IsForAgreement property.
        /// </summary>
        /// <param name="rowid">The rowID.</param>
        /// <param name="value">The checkbox value.</param>
        /// <returns>The action result.</returns>
        [Authorize]
        public ActionResult SetIsForAgreement(int rowid, string value)
        {
            bool oldValueUnCheked = (value == "0");
            SiteDocument SiteDocument = _db.SiteDocuments.Single(u => u.ID == rowid);
            //
            if (SiteDocument != null)
            {
                if (oldValueUnCheked)
                {
                    if (SiteDocument.Culture == null)
                        _db.ExecuteStoreCommand("UPDATE SiteDocuments SET IsForAgreement = NULL WHERE IsSystemDoc IS NULL");
                    else
                        _db.ExecuteStoreCommand(string.Format("UPDATE SiteDocuments SET IsForAgreement = NULL WHERE IsSystemDoc IS NULL AND Culture={0}"), SiteDocument.Culture);
                    //
                    SiteDocument.IsForAgreement = true;
                    SiteDocument.IsNotPublic = null;
                }
                else
                {
                    SiteDocument.IsForAgreement = null;
                }
                //
                try
                {
                    _db.SaveChanges();
                }
                catch (Exception ex)
                {
                    MvcBasicLog.LogException(ex);
                    ModelState.AddModelError("", Resource.SaveInDBError);
                }
            }
            //
            return View("Index");
        }

        /// <summary>
        /// Saves the site document title into the database.
        /// </summary>
        /// <param name="id">The id of the document.</param>
        /// <param name="name">The new name of the document.</param>
        /// <returns>The view result.</returns>
        [Authorize]
        public ViewResult EditSiteDocument(int id, string name)
        {
            SiteDocument SiteDocument = _db.SiteDocuments.Single(u => u.ID == id);
            SiteDocument.Name = name;
            //
            try
            {
                _db.SaveChanges();
            }
            catch (Exception ex)
            {
                MvcBasicLog.LogException(ex);
                ModelState.AddModelError("", Resource.SaveInDBError);
            }
            //
            return View("Index");
        }

        /// <summary>
        /// View a site document into of type TXT/HTML into its own page. 
        /// </summary>
        /// <remarks>
        /// This will be invoked at user request from site main layout.
        /// </remarks>
        /// <param name="id">The id of the document.</param>
        /// <returns>The view result.</returns>
        public ActionResult Details(int id)
        {
            SiteDocument SiteDocument = _db.SiteDocuments.Single(u => u.ID == id);
            if (SiteDocument == null)
                return RedirectToAction("Home", "Index");
            //
            try
            {
                string filePath = (SiteDocument.IsSystemDoc == null && this.Request.IsLocal
                    ? SiteDocument.FileFullName
                    : Path.Combine(HttpContext.Server.MapPath("/Content/Doc"), Path.GetFileName(SiteDocument.FileFullName)));
                //
                SiteDocument.FileData = System.IO.File.ReadAllText(filePath);
            }
            catch (Exception ex)
            {
                MvcBasicLog.LogException(ex);
                ModelState.AddModelError("", ex.Message);
            }
            //
            return View(SiteDocument);
        }

        /// <summary>
        /// Get the given site document as a file result.
        /// </summary>
        /// <remarks>
        /// This will be invoked at user request from site main layout.
        /// </remarks>
        /// <param name="id">The site document ID.</param>
        /// <returns>The view result.</returns>
        public ActionResult GetFileResult(int id)
        {
            SiteDocument SiteDocument = _db.SiteDocuments.FirstOrDefault(d => d.ID == id);
            if (SiteDocument == null)
                return RedirectToAction("Home", "Index");
            //
            OpenFileResult result = new OpenFileResult(SiteDocument.IsSystemDoc == null && this.Request.IsLocal, "\\Content\\Doc");
            result.FileName = SiteDocument.FileFullName;
            result.ContentType = SiteDocument.ContentType;
            //
            return result;
        }
    }
}