﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.ComponentModel.DataAnnotations;

namespace MvcBasic.Logic
{
    
    /// <summary>
    /// Defines the site document entity.
    /// </summary>
    [MetadataType(typeof(SiteDocumentValidation))]
    public partial class SiteDocument
    {
        /// <summary>
        /// Gets or sets the shop document culture.
        /// </summary>
        public SiteCultures SiteDocumentCulture
        {
            get
            {
                return (this.Culture == null ? SiteCultures.All : (SiteCultures)this.Culture);
            }
            set
            {
                this.Culture = (value == SiteCultures.All ? -1 : (int?)value);
            }
        }
        /// <summary>
        /// Gets/Computes Is Editable flag based on file extension.
        /// </summary>
        public bool IsEditable
        {
            get
            {
                if (this.FileFullName == null)
                    return false;
                else
                {
                    string fileName = this.FileFullName.ToLower();
                    return (fileName.EndsWith(".htm")
                        || fileName.EndsWith(".html")
                        || fileName.EndsWith(".txt"));
                }
            }
        }
        /// <summary>
        /// Gets or sets the file data.
        /// </summary>
        /// <remarks>
        /// Used to cache the file data data for TXT or HTML files.
        /// </remarks>
        public string FileData { get; set; }
        /// <summary>
        /// Gets the content type.
        /// </summary>
        public string ContentType
        {
            get
            {
                if (this.FileFullName == null)
                    return null;
                else if (this.FileFullName.ToLower().EndsWith(".pdf"))
                    return "application/pdf";
                else if (this.FileFullName.ToLower().EndsWith(".jpg"))
                    return "image/jpeg";
                else if (this.FileFullName.ToLower().EndsWith(".png"))
                    return "image/png";
                else if (this.IsEditable)
                    return "text/html";
                else
                    return null;
            }
        }

        /// <summary>
        /// Gets all shop documents.
        /// </summary>
        /// <param name="dataContext">The data context.</param>
        /// <returns>All shop documents.</returns>
        public static IQueryable<SiteDocument> GetAllSiteDocuments(MvcBasicSiteEntities dataContext)
        {
            return dataContext.SiteDocuments.AsQueryable();
        }

        /// <summary>
        /// Gets the shop documents that the current user has rights to see.
        /// </summary>
        /// <param name="dataContext">The data context.</param>
        /// <param name="user">The current user.</param>
        /// <param name="culture">The current culture id.</param>
        /// <returns>The shop documents.</returns>
        public static SiteDocument[] GetSiteDocumentsForUser(MvcBasicSiteEntities dataContext, User user, int culture)
        {
            List<SiteDocument> resultsList = (user != null
                ? dataContext.SiteDocuments.Where(d => d.IsForAgreement == null && (d.Culture == -1 || d.Culture == culture)).OrderBy(d => d.ID).ToList()
                : dataContext.SiteDocuments.Where(d => d.IsForAgreement == null && (d.IsNotPublic == null || d.IsNotPublic == false) 
                  && (d.Culture == -1 || d.Culture == culture)).OrderBy(d => d.ID).ToList());
            //
            SiteDocument document = GetShopAgreementDocument(dataContext, culture);
            if (document != null)
                resultsList.Insert(0, document);
            //
            return resultsList.ToArray();
        }

        /// <summary>
        /// Gets the shop agreeement document for the current culture.
        /// </summary>
        /// <param name="dataContext">The data context.</param>
        /// <param name="culture">The current culture id.</param>
        /// <returns>The shop documents.</returns>
        public static SiteDocument GetShopAgreementDocument(MvcBasicSiteEntities dataContext, int culture)
        {
            SiteDocument SiteDocument = dataContext.SiteDocuments.Where(d => d.IsForAgreement == true 
                && d.IsSystemDoc == null && (d.Culture == -1 || d.Culture == culture)).FirstOrDefault();
            if (SiteDocument != null)
                return SiteDocument;
            else
                return dataContext.SiteDocuments.Where(d => d.IsForAgreement == true 
                    && (d.Culture == -1 || d.Culture == culture)).FirstOrDefault();
        }
    }

    /// <summary>
    /// Defines the validation rules and messages for SiteDocument entity.
    /// </summary>
    public class SiteDocumentValidation
    {
        [Required(ErrorMessageResourceType = typeof(Resource), ErrorMessageResourceName = "ValidationRequired")]
        [StringLength(50)]
        public string Name { get; set; }
    }
}

