﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Imaging;

namespace MvcBasic.Logic
{
    /// <summary>
    /// Defines the image utilities class.
    /// </summary>
    /// <remarks>
    /// It is used for image processing.
    /// </remarks>
    public class ImageUtilities
    {
        /// <summary>
        /// Stores the image encoders.
        /// </summary>
        private static Dictionary<string, ImageCodecInfo> _encoders = null;
        /// <summary>
        /// A quick lookup for getting image encoders.
        /// </summary>
        private static Dictionary<string, ImageCodecInfo> Encoders
        {
            get
            {
                //
                // If the quick lookup isn't initialised, initialise it.
                //
                if (_encoders == null)
                {
                    _encoders = new Dictionary<string, ImageCodecInfo>();
                }
                //
                // If there are no codecs, try loading them.
                //
                if (_encoders.Count == 0)
                {
                    foreach (ImageCodecInfo codec in ImageCodecInfo.GetImageEncoders())
                    {
                        //
                        // Add each codec to the quick lookup.
                        //
                        _encoders.Add(codec.MimeType.ToLower(), codec);
                    }
                }
                //
                return _encoders;
            }
        }

        /// <summary>
        /// Resize the image to the specified width and height.
        /// </summary>
        /// <param name="image">The image to resize.</param>
        /// <param name="width">The width to resize to.</param>
        /// <param name="height">The height to resize to.</param>
        /// <returns>The resized image.</returns>
        public static System.Drawing.Bitmap ResizeImage(System.Drawing.Image image, int width, int height)
        {
            Bitmap result = new Bitmap(width, height);
            //
            // Use a graphics object to draw the resized image into the bitmap
            //
            using (Graphics graphics = Graphics.FromImage(result))
            {
                //
                // Set the resize quality modes to high quality.
                //
                graphics.CompositingQuality = System.Drawing.Drawing2D.CompositingQuality.HighQuality;
                graphics.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.HighQualityBicubic;
                graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.HighQuality;
                //
                // Draw the image into the target bitmap.
                //
                graphics.DrawImage(image, 0, 0, result.Width, result.Height);
            }
            //
            // Return the resulting bitmap.
            //
            return result;
        }

        /// <summary> 
        /// Saves an image as a jpeg image file, with the given quality.
        /// </summary> 
        /// <param name="path">Path to which the image would be saved.</param> 
        /// <param name="quality">An integer from 0 to 100, with 100 being the highest quality.</param> 
        /// <exception cref="ArgumentOutOfRangeException">
        /// An invalid value was entered for image quality.
        /// </exception>
        public static void SaveJpeg(string path, Image image, int quality)
        {
            //
            // Ensure the quality is within the correct range
            //
            if ((quality < 0) || (quality > 100))
            {
                //
                // Throw a helpful exception.
                //
                string error = string.Format("Jpeg image quality must be between 0 and 100, with 100 being the highest quality. A value of {0} was specified.", quality);
                throw new ArgumentOutOfRangeException(error);
            }
            //
            // Create an encoder parameter for the image quality.
            //
            EncoderParameter qualityParam = new EncoderParameter(System.Drawing.Imaging.Encoder.Quality, quality);
            ImageCodecInfo jpegCodec = GetEncoderInfo("image/jpeg");
            //
            // Create a collection of all parameters that we will pass to the encoder.
            //
            EncoderParameters encoderParams = new EncoderParameters(1);
            encoderParams.Param[0] = qualityParam;
            //
            // Save the image using the codec and the parameters.
            //
            image.Save(path, jpegCodec, encoderParams);
        }

        /// <summary> 
        /// Returns the image codec with the given mime type.
        /// </summary> 
        public static ImageCodecInfo GetEncoderInfo(string mimeType)
        {
            string lookupKey = mimeType.ToLower();
            ImageCodecInfo foundCodec = null;
            //
            // If we have the encoder, get it to return.
            //
            if (Encoders.ContainsKey(lookupKey))
            {
                //
                // Pull the codec from the lookup
                //
                foundCodec = Encoders[lookupKey];
            }
            //
            return foundCodec;
        }

        /// <summary> 
        /// Saves an image as a jpeg image into the given memory stream, with the given quality.
        /// </summary> 
        /// <param name="ms">The memory stream.</param> 
        /// <param name="quality">An integer from 0 to 100, with 100 being the highest quality.</param> 
        /// <param name="image">The image to save.</param>
        /// <exception cref="ArgumentOutOfRangeException">
        /// An invalid value was entered for image quality.
        /// </exception>
        /// <returns>The stream with the image in JPEG format ready to be used.</returns>
        public static System.IO.Stream SaveJpeg(System.IO.MemoryStream ms, Bitmap image, int quality)
        {
            //
            // Ensure the quality is within the correct range.
            //
            if ((quality < 0) || (quality > 100))
            {
                //
                // Throw a helpful exception.
                //
                string error = string.Format("Jpeg image quality must be between 0 and 100, with 100 being the highest quality. A value of {0} was specified.", quality);
                throw new ArgumentOutOfRangeException(error);
            }
            //
            // Create an encoder parameter for the image quality.
            //
            EncoderParameter qualityParam = new EncoderParameter(System.Drawing.Imaging.Encoder.Quality, quality);
            ImageCodecInfo jpegCodec = GetEncoderInfo("image/jpeg");
            //
            // Create a collection of all parameters that we will pass to the encoder
            //
            EncoderParameters encoderParams = new EncoderParameters(1);
            encoderParams.Param[0] = qualityParam;
            //
            // Save the image using the codec and the parameters into the stream.
            //
            image.Save(ms, jpegCodec, encoderParams);
            return ms;
        }
    }
}

