<?php

/**
 * Converts data to safety format
 * _plain($data, $length, $required)
 * _date($data, $min_date, $max_date, $date_type, $required)
 * _email($data, $length, $required)
 * _int($data, $max_value, $required)
 * _float($data, $precision, $max_value, $required)
 * _filename($data, $lowercase)
 * _url($data)
 * 
 * $Id$
 * @package system
 */
class DataFormat 
{
	/**
	 * explains whether the data is required
	 */
	const REQUIRED = true;
	
	/**
	 * character map for conversion 
	 */
	static $CHARMAP = array(
		'ö' => 'o',
		'Ö' => 'O',
		'ó' => 'o',
		'Ó' => 'O',
		'ő' => 'o',
		'Ő' => 'O',
		'ú' => 'u',
		'Ú' => 'U',
		'ű' => 'u',
		'Ű' => 'U',
		'ü' => 'u',
		'Ü' => 'U',
		'á' => 'a',
		'Á' => 'A',
		'é' => 'e',
		'É' => 'E',
		'í' => 'i',
		'Í' => 'I',
		' ' => '_'
	);
	
	
	/**
	 * strips the given data and returns as a plain text
	 * @param string $data
	 * @param int $length
	 * @param bool $required
	 * @return bool|string 
	 */
	public static function _plain($data, $length = 0, $required = false)
	{
		$limit = $length > 0 ? $length : strlen($data);
		$data = substr(trim(stripslashes(strip_tags($data))),0,$limit);
		
		if($required === self::REQUIRED and empty($data))
		{
			return null;
		}
		return (string)$data;
	}
	
	/**
	 * Validates data as a date and optionally checks if it is in a [mindate , maxdate] interval
	 * @param mixed $data - int if timestamp, string if date string, array if exploded string
	 *        Note: When data is a string, the function only applies strtotime() compatible string.
	 *              If the number of the year is specified in a two digit format, the values between 
	 *              0-69 are mapped to 2000-2069 and 70-100 to 1970-2000.
	 * @param int $mindate 
	 * @param int $maxdate
	 * @param string $date_type - gives the type of the $min_date and $max_date. Default is the timestamp
	 *        given as 'second'. 
	 * @param bool $required
	 * @return bool|int - timestamp
	 */
	public static function _date($data, $min_date = 0, $max_date = 0, $date_type = 'second', $required = false)
	{
		if(is_string($data))
		{
			$data = strtotime($data);
		}
		
		if(is_array($data))
		{
			$year = isset($data[0]) ? (int)$data[0] : -1;
			$month = isset($data[1]) ? (int)$data[1] : -1;
			$day = isset($data[2]) ? (int)$data[2] : -1;
			$hour = (isset($data[3]) and is_numeric($data[3]) and 24 >= $data[3]) ? (int)$data[3] : 0;
			$minute = (isset($data[4]) and is_numeric($data[4]) and 59 >= $data[4]) ? (int)$data[4] : 0;
			$second = (isset($data[5]) and is_numeric($data[5]) and 59 >= $data[5]) ? (int)$data[5] : 0;
			
			if(checkdate($month, $day, $year) !== false)
			{
				$data = mktime($hour, $minute, $second, $month, $day, $year);
				
				if((int)$min_date > 0 or (int)$max_date > 0)
				{
					$mn_date = time();
					$mx_date = time();
					
					if(strcmp($date_type, 'year') == 0)
					{
						$mn_date = mktime(date('H'),date('i'), date('s'), date('m'), date('d'), date('Y') - (int)$min_date);
						$mx_date = mktime(date('H'),date('i'), date('s'), date('m'), date('d'), date('Y') - (int)$max_date);
					}	
					elseif(strcmp($date_type, 'month') == 0)
					{
						$mn_date = mktime(date('H'),date('i'), date('s'), date('m') - (int)$min_date, date('d'), date('Y'));
						$mx_date = mktime(date('H'),date('i'), date('s'), date('m') - (int)$max_date, date('d'), date('Y'));
					}
					elseif(strcmp($date_type, 'day') == 0)
					{
						$mn_date = time() - ((int)$min_date * 60 * 60 * 24);
						$mx_date = time() - ((int)$max_date * 60 * 60 * 24);
					}
					elseif(strcmp($date_type, 'hour') == 0)
					{
						$mn_date = time() - ((int)$min_date * 60 * 60);
						$mx_date = time() - ((int)$max_date * 60 * 60);
					}
					elseif(strcmp($date_type, 'minute') == 0)
					{
						$mn_date = time() - ((int)$min_date * 60);
						$mx_date = time() - ((int)$max_date * 60);
					}
					elseif(strcmp($date_type, 'second') == 0)
					{
						$mn_date = time() - (int)$min_date;
						$mx_date = time() - (int)$max_date;
					}
					
					$min_date = min($mn_date, $mx_date);
					$max_date = max($mn_date, $mx_date);
					
					if(!($min_date <= $data and $data <= $max_date))
					{
						$data = false;
					}
				}
			}
		}
		
		if(!is_int($data))
		{
			$data = false;
		}	

		if($required === self::REQUIRED and $data === false)
		{
			return null;
		}
		return (int)$data;
	}
	
	/**
	 * Validates data as an e-mail address: checking format and length
	 * @param string $data - e-mail address
	 * @param int $length - max length of the address
	 * @param bool $required
	 * @return bool|string - email
	 */
	public static function _email($data, $length = 0, $required = false)
	{
		$email = trim($data);
		$data = false;
		$limit = $length > 0 ? $length : 256;
		if(strlen($email) < $limit and preg_match('/^([a-zA-Z0-9\_\-\.]+)@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.)|(([a-zA-Z0-9\-]+\.)+))([a-zA-Z]{2,4}|[0-9]{1,3})(\]?)$/',$email))
		{
	  	$data = $email;
	  }
		
		if($required === self::REQUIRED and $data === false)
		{
			return null;
		}
		return (string)$data;
	}
	
	/**
	 * Validates data as intiger
	 * @param string $data - the number
	 * @param int $max_value
	 * @param bool $required
	 * @return bool|int 
	 */
	public static function _int($data, $max_value = 0, $required = false)
	{
		if(is_numeric($data))
		{
			$int = intval(preg_replace('/[^0-9]/i','',$data));
			$data = false;
			
			if($max_value == 0 or ($max_value > 0 and $max_value >= $int))
			{
				$data = $int;
			}
		}
		else
		{
			$data = false;
		}
		
		if($required === self::REQUIRED and $data === false)
		{
			return null;
		}
		return (int)$data;
	}
	
	/**
	 * validates data as a floating point number
	 * @param mixed $data - the number represented in a string or in a numeric type
	 * @param int $precision - number of digits after the decimal point
	 * @param int $max_value
	 */
	public static function _float($data, $precision = 2, $max_value = 0.0, $required = false)
	{
		if(is_numeric($data))
		{
			$float = strval($data);
			$data = false; 
				
			$float = preg_replace('@([:space:])+@','',$float);
			
			// like 1,234,567 or 1,234,567.89
			if(substr_count($float, ',') > 1 and substr_count($float, '.') >= 1) 
			{
				$float = floatval(str_replace(',','',$float));											
			}
			// like 1,234.56
			else if(substr_count($float, ',') == 1 and substr_count($float, '.') == 1) 
			{
				$float = floatval(str_replace(',','',$float));					
			}				
			// like 1,234 or 123,45
			else if(substr_count($float, ',') == 1 and substr_count($float, '.') == 0) 
			{
				$pos = strlen($float) - strrpos($float,',');
				
				// like 1,234 = 1234
				if($pos >= 4 or $pos <= 1)  
				{
					$float = floatval(str_replace(',','',$float));						
				}
				// like 123,4 -> 123.4
				else if($pos < 4 and $pos > 1) 
				{
					$float = floatval(str_replace(',','.',$float));						
				}					
			}
			// any other case
			else 
			{
				$float = floatval(preg_replace('/[^0-9]/','',$float));
			}
			
			if($max_value == 0.0 or ($max_value > 0.0 and $max_value >= $float))
			{
				$data = round($float, $precision);
			}
		}
		else
		{
			$data = false;
		}
		
		if($required === self::REQUIRED and $data === false)
		{
			return null;
		}
		return (float)$data;
	}
	
	/**
	 * validates and converts data as a string suitable for file name
	 * @param string $data
	 * @param bool $lowercase
	 * @return string
	 */
	public static function _filename($data, $lowercase = true)
	{
		$data = DataFormat::_plain($data, 128);
		$data = strtr($data, self::$CHARMAP);
		if($lowercase)
		{
			$data = strtolower($data);			
		}
		
		$data = preg_replace('/[^a-zA-Z0-9\_\.]/','',$data);
		
		return $data;
	}
	
	/**
	 * convert string to url
	 */
	public function _url($data, $lowercase = true)
  {
		$data = trim($data);
			
		$data = preg_replace('@&amp;@i', 'es', $data);
		$data = preg_replace('@(&|\+)@i', 'es', $data);
		$data = preg_replace('@\s@i', '_', $data);
		$data = preg_replace('@(í|Í)@i', 'i', $data);        
		$data = preg_replace('@(é|É)@i', 'e', $data);
		$data = preg_replace('@(á|Á)@i', 'a', $data);				
		$data = preg_replace('@(ó|ö|ő|Ó|Ö|Ő)@i', 'o', $data);				
		$data = preg_replace('@(ú|ü|ű|Ú|Ü|Ű)@i', 'u', $data);				
		
		$data = trim(preg_replace('/[^a-zA-Z0-9\_\-\.]/', ' ', $data));
  	if($lowercase)
		{
			$data = strtolower($data);			
		}				
		$data = preg_replace('/\_$/', '', $data);
	
    return preg_replace('/ /', '_', $data);
  }
}
