﻿using MvcSchedule.Objects;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Text;
using System.Threading.Tasks;
using System.Web.Mvc;
using System.IO;
using System.Web.UI;

namespace MvcSchedule
{

    /// <summary>
    /// Extension HTML helpers for the MvcSchedule components.
    /// </summary>
    public static class ScheduleHelpers
    {

        /// <summary>
        /// Render a general schedule using a display expression for the items
        /// </summary>
        /// <typeparam name="M">Model type</typeparam>
        /// <typeparam name="TItem">Model item type</typeparam>
        /// <param name="htmlHelper">HTML helper</param>
        /// <param name="model">The data</param>
        /// <param name="dataRangeStartExpression">Expression that yields the start of the event</param>
        /// <param name="dataRangeEndExpression">Expression that yields the end of the event</param>
        /// <param name="titleExpression">Expression that yields the title of the event</param>
        /// <param name="itemDisplayExpression">Expression that yields the way to display an item on the schedule</param>
        /// <param name="options">Options</param>
        /// <param name="dataRangeDisplayExpression">Expression that yields the way to display the start or the end of the event</param>
        /// <param name="titleDisplayExpression">Expression that yields the way to display the title of the event</param>
        /// <param name="dateHeaderDisplayExpression">Expression that yields the date to display the date header of the event. Only used when Options.SeparateDateHeader==true</param>
        /// <param name="emptyDataText">Text shown when there's no data to display</param>
        /// <returns>Html string</returns>
        public static MvcHtmlString ScheduleGeneralFor<M, TItem>(this HtmlHelper<M> htmlHelper, 
            IEnumerable<TItem> model,
            Expression<Func<TItem, object>> dataRangeStartExpression,
            Expression<Func<TItem, object>> dataRangeEndExpression,
            Expression<Func<TItem, object>> titleExpression,
            Expression<Func<TItem, string>> itemDisplayExpression,
            MvcScheduleGeneralOptions options = null,
            Expression<Func<object, string>> dataRangeDisplayExpression = null,
            Expression<Func<object, string>> titleDisplayExpression = null,
            Expression<Func<DateTime, string>> dateHeaderDisplayExpression = null,
            string emptyDataText = null)
        {
            var sch = new ScheduleGeneral<TItem>();
            sch.DataRangeStartExpression = dataRangeStartExpression;
            sch.DataRangeEndExpression = dataRangeEndExpression;
            sch.TitleExpression = titleExpression;
            sch.ItemDisplayExpression = itemDisplayExpression;
            sch.DataRangeDisplayExpression = dataRangeDisplayExpression;
            sch.TitleDisplayExpression = titleDisplayExpression;
            sch.DateHeaderDisplayExpression = dateHeaderDisplayExpression;
            sch.EmptyDataText = emptyDataText;
            sch.Options = options;

            return MvcHtmlString.Create(sch.RenderTable(model, htmlHelper));
        }

        /// <summary>
        /// Render a general schedule using a partial view for the items
        /// </summary>
        /// <typeparam name="M">Model type</typeparam>
        /// <typeparam name="TItem">Model item type</typeparam>
        /// <param name="htmlHelper">HTML helper</param>
        /// <param name="model">The data</param>
        /// <param name="dataRangeStartExpression">Expression that yields the start of the event</param>
        /// <param name="dataRangeEndExpression">Expression that yields the end of the event</param>
        /// <param name="titleExpression">Expression that yields the title of the event</param>
        /// <param name="itemPartialViewName">Name of the partial view that displays the item on the schedule</param>
        /// <param name="options">Options</param>
        /// <param name="dataRangeDisplayExpression">Expression that yields the way to display the start or the end of the event</param>
        /// <param name="titleDisplayExpression">Expression that yields the way to display the title of the event</param>
        /// <param name="dateHeaderDisplayExpression">Expression that yields the date to display the date header of the event. Only used when Options.SeparateDateHeader==true</param>
        /// <param name="emptyDataText">Text shown when there's no data to display</param>
        /// <returns>Html string</returns>
        public static MvcHtmlString ScheduleGeneralFor<M, TItem>(this HtmlHelper<M> htmlHelper,
            IEnumerable<TItem> model,
            Expression<Func<TItem, object>> dataRangeStartExpression,
            Expression<Func<TItem, object>> dataRangeEndExpression,
            Expression<Func<TItem, object>> titleExpression,
            string itemPartialViewName,
            MvcScheduleGeneralOptions options = null,
            Expression<Func<object, string>> dataRangeDisplayExpression = null,
            Expression<Func<object, string>> titleDisplayExpression = null,
            Expression<Func<DateTime, string>> dateHeaderDisplayExpression = null,
            string emptyDataText = null)
        {
            var sch = new ScheduleGeneral<TItem>();
            sch.DataRangeStartExpression = dataRangeStartExpression;
            sch.DataRangeEndExpression = dataRangeEndExpression;
            sch.TitleExpression = titleExpression;
            sch.ItemPartialViewName = itemPartialViewName;
            sch.DataRangeDisplayExpression = dataRangeDisplayExpression;
            sch.TitleDisplayExpression = titleDisplayExpression;
            sch.DateHeaderDisplayExpression = dateHeaderDisplayExpression;
            sch.EmptyDataText = emptyDataText;
            sch.Options = options;

            return MvcHtmlString.Create(sch.RenderTable(model, htmlHelper));
        }

        /// <summary>
        /// Render a calendar schedule using a display expression for the items
        /// </summary>
        /// <typeparam name="M">Model type</typeparam>
        /// <typeparam name="TItem">Model item type</typeparam>
        /// <param name="htmlHelper">HTML helper</param>
        /// <param name="model">The data</param>
        /// <param name="startTimeExpression">Expression that yields the start time of the event</param>
        /// <param name="endTimeExpression">Expression that yields the end time of the event</param>
        /// <param name="dateExpression">Expression that yields the date of the event. 
        /// Ignored when TimeExpressionsContainDate==true. 
        /// When TimeExpressionsContainDate==false, this should yield a result of type Date</param>
        /// <param name="itemDisplayExpression">Expression that yields the way to display the item on the schedule</param>
        /// <param name="options">Options</param>
        /// <param name="timeDisplayExpression">Expression that yields the way to display the time of the event</param>
        /// <param name="dateDisplayExpression">Expression that yields the way to display the date of the event</param>
        /// <param name="emptyDataText">Text shown when there's no data to display</param>
        /// <returns>Html string</returns>
        public static MvcHtmlString ScheduleCalendarFor<M, TItem>(this HtmlHelper<M> htmlHelper,
            IEnumerable<TItem> model,
            Expression<Func<TItem, object>> startTimeExpression,
            Expression<Func<TItem, object>> endTimeExpression,
            Expression<Func<TItem, object>> dateExpression,
            Expression<Func<TItem, string>> itemDisplayExpression,
            MvcScheduleCalendarOptions options = null,
            Expression<Func<object, string>> timeDisplayExpression = null,
            Expression<Func<object, string>> dateDisplayExpression = null,
            string emptyDataText = null)
        {
            var sch = new ScheduleCalendar<TItem>();
            sch.StartTimeExpression = startTimeExpression;
            sch.EndTimeExpression = endTimeExpression;
            sch.DateExpression = dateExpression;
            sch.ItemDisplayExpression = itemDisplayExpression;
            sch.TimeDisplayExpression = timeDisplayExpression;
            sch.DateDisplayExpression = dateDisplayExpression;
            sch.EmptyDataText = emptyDataText;
            sch.Options = options;
            
            return MvcHtmlString.Create(sch.RenderTable(model, htmlHelper));
        }

        /// <summary>
        /// Render a calendar schedule using a partial view for the items
        /// </summary>
        /// <typeparam name="M">Model type</typeparam>
        /// <typeparam name="TItem">Model item type</typeparam>
        /// <param name="htmlHelper">HTML helper</param>
        /// <param name="model">The data</param>
        /// <param name="startTimeExpression">Expression that yields the start time of the event</param>
        /// <param name="endTimeExpression">Expression that yields the end time of the event</param>
        /// <param name="dateExpression">Expression that yields the date of the event. 
        /// Ignored when TimeExpressionsContainDate==true. 
        /// When TimeExpressionsContainDate==false, this should yield a result of type Date</param>
        /// <param name="itemPartialViewName">Name of the partial view that displays the item on the schedule</param>
        /// <param name="options">Options</param>
        /// <param name="timeDisplayExpression">Expression that yields the way to display the time of the event</param>
        /// <param name="dateDisplayExpression">Expression that yields the way to display the date of the event</param>
        /// <param name="emptyDataText">Text shown when there's no data to display</param>
        /// <returns>Html string</returns>
        public static MvcHtmlString ScheduleCalendarFor<M, TItem>(this HtmlHelper<M> htmlHelper,
            IEnumerable<TItem> model,
            Expression<Func<TItem, object>> startTimeExpression,
            Expression<Func<TItem, object>> endTimeExpression,
            Expression<Func<TItem, object>> dateExpression,
            string itemPartialViewName,
            MvcScheduleCalendarOptions options = null,
            Expression<Func<object, string>> timeDisplayExpression = null,
            Expression<Func<object, string>> dateDisplayExpression = null,
            string emptyDataText = null)
        {
            var sch = new ScheduleCalendar<TItem>();
            sch.StartTimeExpression = startTimeExpression;
            sch.EndTimeExpression = endTimeExpression;
            sch.DateExpression = dateExpression;
            sch.ItemPartialViewName = itemPartialViewName;
            sch.TimeDisplayExpression = timeDisplayExpression;
            sch.DateDisplayExpression = dateDisplayExpression;
            sch.EmptyDataText = emptyDataText;
            sch.Options = options;

            return MvcHtmlString.Create(sch.RenderTable(model, htmlHelper));
        }

    }

}
