﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;

namespace MvcSchedule.Objects
{

    /// <summary>
    /// Extension of the ScheduleTable class to represent a table with value marks.
    /// </summary>
    internal class ScheduleTableWithValueMarks : ScheduleTable
    {

        #region Public properties

        public override int RowsPerValue
        {
            get { return 2; }
        }

        #endregion

        #region Constructor and initialization

        public ScheduleTableWithValueMarks(int repetitionCount)
            : base(repetitionCount)
        {
        }

        #endregion

        #region Public methods

        public override void Initialize(int rowCount, string titleCss, string rangeHeaderCss, string backgroundCss)
        {
            //MyBase.Initialize(weeks, rowCount, titleStyle, rangeHeaderStyle, backgroundStyle)
            for (int iWeek = 1; iWeek <= _repetitionCount; iWeek++)
            {
                // Add a row of title header cells
                this.AddRow();
                for (int column = 0; column < ColumnCount; column++)
                {
                    ScheduleTableCell cell = this.Cell(this.RowCount - 1, column);
                    cell.IsHeader = true;
                    cell.CssClass = titleCss;
                    cell.Text = "&nbsp;";
                }

                for (int row = 1; row <= rowCount; row++)
                {
                    this.AddRow(); // add a cell for the header column
                    ScheduleTableCell headerCell = this.Cell(this.RowCount - 1, 0);
                    headerCell.IsHeader = true;
                    headerCell.CssClass = rangeHeaderCss;
                    headerCell.Text = "&nbsp;";
                    headerCell.RowSpan = 2;

                    if (row == 1)
                    {
                        for (int column = 1; column < ColumnCount; column++)
                        {
                            ScheduleTableCell cell = this.Cell(this.RowCount - 1, column);
                            cell.CssClass = backgroundCss;
                            cell.Text = "&nbsp;";
                        }
                    }
                    // Add an extra row. The cells in this row will span 2 rows (except for the last row).
                    this.AddRow();
                    // no need to set anything for the first cell in this row, it will be spanned anyway
                    for (int column = 1; column < ColumnCount; column++)
                    {
                        ScheduleTableCell cell = this.Cell(this.RowCount - 1, column);
                        cell.CssClass = backgroundCss;
                        cell.Text = "&nbsp;";
                        if (row < rowCount)
                        {
                            // the last row only has a span of 1
                            cell.RowSpan = 2;
                        }
                    }
                }
            }
        }

        public override void InsertColumn(int index, string style)
        {
            base.InsertColumn(index, style);

            int rowCountPerWeek = this.RowCount / _repetitionCount;
            for (int week = 0; week <= _repetitionCount - 1; week++)
            {
                for (int row = 0; row <= rowCountPerWeek - 1; row++)
                {
                    ScheduleTableCell newCell = this.Cell(week * rowCountPerWeek + row, index);
                    if (row % 2 == 0 && row > 0 && row < rowCountPerWeek - 1)
                    {
                        newCell.RowSpan = 2;
                    }
                }
            }
        }

        /// <summary>
        /// Add range value marks to indicate the range values
        /// </summary>
        public override void AddRangeValueMarks(string titleStyle, string rangeheaderStyle)
        {
            // Insert an extra column for range marks after the first column
            InsertColumn(1, rangeheaderStyle);

            int rowCountPerWeek = this.RowCount / _repetitionCount;
            for (int week = 0; week < _repetitionCount; week++)
            {
                // First set the cell in the title row
                int row = week * rowCountPerWeek;
                ScheduleTableCell titleCell = Cell(row, 1);
                titleCell.IsHeader = true;
                titleCell.Text = "&nbsp;";
                titleCell.CssClass = titleStyle;

                for (int rowInThisWeek = 1; rowInThisWeek <= rowCountPerWeek - 1; rowInThisWeek++)
                {
                    row = rowInThisWeek + week * rowCountPerWeek;

                    ScheduleTableCell markerCell = Cell(row, 1);
                    markerCell.Text = "&nbsp;";
                    markerCell.RowSpan = 1; // InsertColumn may have set 2, but it should be 1
                    markerCell.BorderLeft = false;

                    ScheduleTableCell headerCell = Cell(row, 0);
                    if (rowInThisWeek == 1)
                    {
                        // first row in this week
                        headerCell.BorderBottom = false;
                        headerCell.BorderTop = false;
                        headerCell.BorderRight = false;

                        markerCell.BorderRight = false;
                        markerCell.BorderTop = false;
                        markerCell.BorderBottom = true;
                    }
                    else if (rowInThisWeek == rowCountPerWeek - 1)
                    {
                        // last row in this week
                        headerCell.BorderTop = false;

                        markerCell.BorderRight = false;
                        markerCell.BorderTop = true;
                        markerCell.BorderBottom = false;
                    }
                    else
                    {
                        // other rows
                        headerCell.BorderBottom = false;
                        headerCell.BorderTop = false;
                        headerCell.BorderRight = false;

                        if (rowInThisWeek % 2 == 1)
                        {
                            markerCell.BorderTop = false;
                            markerCell.BorderBottom = true;
                        }
                        else
                        {
                            markerCell.BorderTop = true;
                            markerCell.BorderBottom = false;
                        }
                    }
                    markerCell.CssClass = rangeheaderStyle;
                }
            }
        }

        #endregion

    }

}