﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;

namespace MvcSchedule.Objects
{

    /// <summary>
    /// Table representing the contents of a schedule control.
    /// For simplicity, the table is always built in vertical layout, and only converted to a horizontal web table when necessary.
    /// </summary>
    internal class ScheduleTable
    {

        private List<List<ScheduleTableCell>> _cells = new List<List<ScheduleTableCell>>();
        private int _columnCount = 0;

        protected int _repetitionCount = 1;

        #region Public properties

        public int RowCount
        {
            get { return _cells.Count; }
        }

        public int ColumnCount
        {
            get { return _columnCount; }
        }

        public ScheduleTableCell Cell(int row, int column)
        {
            if (row < 0 || row > RowCount - 1)
            {
                throw new MvcScheduleException("row out of range");
            }
            if (column < 0 || column > ColumnCount - 1)
            {
                throw new MvcScheduleException("column out of range");
            }
            return _cells[row][column];
        }

        public virtual int RowsPerValue
        {
            get { return 1; }
        }

        #endregion

        #region Constructor and initialization

        public ScheduleTable(int repetitionCount)
        {
            this._repetitionCount = repetitionCount;
        }

        #endregion

        #region Public methods

        public virtual void Initialize(int rowCount, string titleCss, string rangeHeaderCss, string backgroundCss)
        {
            for (int iWeek = 1; iWeek <= _repetitionCount; iWeek++)
            {
                
                this.AddRow(); // Add a row of title header cells
                for (int column = 0; column < ColumnCount; column++)
                {
                    ScheduleTableCell cell = this.Cell(this.RowCount - 1, column);
                    cell.IsHeader = true;
                    cell.CssClass = titleCss;
                    cell.Text = "&nbsp;";
                }

                for (int row = 1; row <= rowCount; row++)
                {
                    this.AddRow();
                    // add a cell for the header column
                    ScheduleTableCell headerCell = this.Cell(this.RowCount - 1, 0);
                    headerCell.IsHeader = true;
                    headerCell.CssClass = rangeHeaderCss;
                    headerCell.Text = "&nbsp;";

                    for (int column = 1; column < ColumnCount; column++)
                    {
                        ScheduleTableCell cell = this.Cell(this.RowCount - 1, column);
                        cell.CssClass = backgroundCss;
                        cell.Text = "&nbsp;";
                    }
                }
            }
        }

        public void AddColumn()
        {
            foreach (List<ScheduleTableCell> row in _cells)
            {
                row.Add(new ScheduleTableCell());
            }
            _columnCount += 1;
        }

        public virtual void InsertColumn(int column, string cssClass)
        {
            if (column < 0 || column > ColumnCount - 1)
            {
                throw new MvcScheduleException("Index out of range");
            }
            foreach (List<ScheduleTableCell> rowOfCells in _cells)
            {
                ScheduleTableCell cell = new ScheduleTableCell();
                cell.ValueIndex = rowOfCells[column].ValueIndex;
                cell.TitleIndex = column;
                cell.CssClass = cssClass;
                cell.Text = "&nbsp;";
                rowOfCells.Insert(column, cell);
            }
            _columnCount += 1;
        }

        public void AddRow()
        {
            List<ScheduleTableCell> row = new List<ScheduleTableCell>();
            for (int column = 0; column < ColumnCount; column++)
            {
                ScheduleTableCell cell = new ScheduleTableCell();
                cell.ValueIndex = RowCount;
                cell.TitleIndex = column;
                row.Add(cell);
            }
            _cells.Add(row);
        }

        public void SwapCells(int row, int column1, int column2)
        {
            ScheduleTableCell temp = _cells[row][column2];
            _cells[row][column2] = _cells[row][column1];
            _cells[row][column1] = temp;
        }

        public void HideSpannedCells()
        {
            for (int row = 0; row < RowCount; row++)
            {
                for (int column = 0; column < ColumnCount; column++)
                {
                    ScheduleTableCell cell = this.Cell(row, column);
                    if (!cell.Visible)
                        continue;
                    for (int column2 = column; column2 <= column + cell.ColumnSpan - 1; column2++)
                    {
                        for (int row2 = row; row2 <= row + cell.RowSpan - 1; row2++)
                        {
                            if (row2 == row && column2 == column)
                                continue;
                            this.Cell(row2, column2).Visible = false;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Add range value marks to indicate the range values
        /// Overridden in derived class ScheduleTableWithValueMarks.
        /// </summary>
        public virtual void AddRangeValueMarks(string titleStyle, string rangeheaderStyle)
        {
            // nothing to do
        }

        /// <summary>
        /// The number of rows per repetition
        /// </summary>
        /// <param name="valueCount">The number of values</param>
        /// <returns>Integer containing the number of rows</returns>
        public int RowsPerRepetition(int valueCount)
        {
            return valueCount * RowsPerValue + 1;
        }

        #endregion

    }

}

