﻿using MvcSchedule.Data;
using MvcSchedule.Objects;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.ComponentModel;
using System.Linq.Expressions;
using System.Text;
using System.Web.Mvc;
using System.Web.UI;

namespace MvcSchedule.Objects
{


    // -------------------------- MvcSchedule Controls v1.0.0.0 -----------------------
    //
    // ------------------------------ www.rekenwonder.com ---------------------------------
    //
    // ----------- Licensing: These controls are free under the LGPL license. -------------
    // ----------- See: http://www.gnu.org/licenses/lgpl.html for more details. -----------

    /// -----------------------------------------------------------------------------
    /// Project	 : MvcSchedule
    /// Class	 : ScheduleGeneral
    ///
    /// -----------------------------------------------------------------------------
    /// <summary>
    /// The ScheduleGeneral control is designed to represent a schedule in a general format.
    /// </summary>
    /// -----------------------------------------------------------------------------
    internal class ScheduleGeneral<TItem> : BaseSchedule<TItem>
    {

        #region Private and protected properties

        private MvcScheduleGeneralOptions _options;
        private ArrayList _arrTitleValues;

        private Expression<Func<DateTime, string>> _dateHeaderDisplayExpression;

        public MvcScheduleGeneralOptions Options
        {
            get { return _options; }
            set { _options = value; }
        }

        protected override MvcScheduleOptions BaseOptions
        {
            get { return Options; }
        }

        /// <summary>
        /// Only used when Options.SeparateDateHeader==true
        /// </summary>
        public Expression<Func<DateTime, string>> DateHeaderDisplayExpression
        {
            get { return _dateHeaderDisplayExpression; }
            set { _dateHeaderDisplayExpression = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// list with values to be shown in title header
        /// </summary>
        /// -----------------------------------------------------------------------------
        protected ArrayList ArrTitleValues
        {
            get { return _arrTitleValues; }
            set { _arrTitleValues = value; }
        }

        #endregion

        #region Methods

        // Check if all properties are set to make the control work
        protected internal override void CheckConfiguration()
        {
            base.CheckConfiguration();
            if (TitleExpression == null)
                throw new MvcScheduleException("The TitleExpression is not set");
            if (DataRangeStartExpression == null)
                throw new MvcScheduleException("The DataRangeStartExpression is not set");
            if (DataRangeEndExpression == null)
                throw new MvcScheduleException("The DataRangeEndExpression is not set");
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// create the list with all range header values (Start or End)
        /// </summary>
        /// <param name="dataList"></param>
        /// -----------------------------------------------------------------------------
        protected override void FillRangeValueArray(ScheduleItemDataList<TItem> dataList)
        {
            ArrRangeValues = new ArrayList();
            if (BaseOptions.FullTimeScale)
            {
                TimeSpan tsInc = new TimeSpan(0, Options.TimeScaleInterval, 0);
                if (dataList.Count == 0) return; 

                // find the first start time and the last end time in the list
                DateTime firstStartTime = DateTime.MaxValue;
                DateTime lastEndTime = DateTime.MinValue;
                foreach (ScheduleItemData<TItem> item in dataList)
                {
                    object val1 = item.StartValue;
                    object val2 = item.EndValue;
                    if (val1 == null)
                        continue;
                    // Nulls are allowed (for creating titles without content) but will not show up
                    if (val1 is DBNull)
                        continue;
                    DateTime dt1 = (DateTime)val1;
                    if (dt1 < firstStartTime)
                        firstStartTime = dt1;
                    DateTime dt2 = (DateTime)val2;
                    if (dt2 > lastEndTime)
                        lastEndTime = dt2;
                }
                // add incrementing times to the array
                DateTime dt = firstStartTime;
                while (dt <= lastEndTime)
                {
                    TimeSpan t = BaseOptions.StartOfTimeScale;
                    while (TimeSpan.Compare(t, BaseOptions.EndOfTimeScale) < 0)
                    {
                        DateTime dti = new DateTime(dt.Year, dt.Month, dt.Day, t.Hours, t.Minutes, 0);
                        ArrRangeValues.Add(dti);
                        t = t.Add(tsInc);
                    }
                    // Add the end of the timescale as well to make sure it's there
                    // e.g. in the case of EndOfTimeScale=23:59 and TimeScaleInterval=1440, this is imperative
                    DateTime dtEnd = new DateTime(dt.Year, dt.Month, dt.Day, BaseOptions.EndOfTimeScale.Hours, BaseOptions.EndOfTimeScale.Minutes, 0);
                    ArrRangeValues.Add(dtEnd);
                    dt = dt.AddDays(1);
                }
                
            }
            else // ! FullTimeScale
            {
                // Just add the times from the data list
                foreach (ScheduleItemData<TItem> item in dataList)
                {                    
                    object val = item.StartValue;
                    if (val != null && !(val is DBNull)) // Nulls are allowed (for creating titles without content) but will not show up
                    {
                        ArrRangeValues.Add(val);
                        ArrRangeValues.Add(item.EndValue);
                    }
                }
            }
            ArrRangeValues.Sort();
            ArrRangeValues.RemoveDoubles();
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// create the list with all titles
        /// </summary>
        /// <param name="dataList"></param>
        /// -----------------------------------------------------------------------------
        protected override void FillTitleValueArray(ScheduleItemDataList<TItem> dataList)
        {
            ArrTitleValues = new ArrayList();
            foreach (ScheduleItemData<TItem> item in dataList)
            {
                ArrTitleValues.Add(item.TitleValue);
            }
            if (Options.AutoSortTitles)
            {
                ArrTitleValues.Sort();
            }
            ArrTitleValues.RemoveDoubles();
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Add date headers to the table when SeparateDateHeader=True
        /// </summary>
        /// -----------------------------------------------------------------------------
        protected override void AddDateHeaderData()
        {
            if (!Options.SeparateDateHeader) return;

            // merge all cells having the same date in the first (date) header
            if (ArrRangeValues.Count == 0)  return;

            if (!(ArrRangeValues[0] is DateTime))
            {
                throw new MvcScheduleException("If SeparateDateHeader is true, then DataRangeStartField and DataRangeEndField should be of type DateTime");
            }
            ScheduleTable.InsertColumn(0, Options.RangeHeaderCss);

            ScheduleTable.Cell(0, 0).CssClass = Options.TitleCss;
            ScheduleTable.Cell(0, 0).IsHeader = true;

            DateTime prevRangeValue = Convert.ToDateTime(ArrRangeValues[0]);
            int numberOfValuesForCurrentDate = 0;

            for (int index = 0; index < ArrRangeValues.Count; index++)
            {
                DateTime rangeValue = Convert.ToDateTime(ArrRangeValues[index]);
                if (rangeValue.Date != prevRangeValue.Date)
                {
                    // this value has another date than the previous one (the one above)
                    // make the cell above span the cells that have the same date
                    FillDateCell(index - numberOfValuesForCurrentDate, numberOfValuesForCurrentDate);
                    prevRangeValue = rangeValue;

                    numberOfValuesForCurrentDate = 1;
                }
                else
                {
                    numberOfValuesForCurrentDate ++;
                }
            }
            // finish by adding the last date 
            FillDateCell(ArrRangeValues.Count - numberOfValuesForCurrentDate, numberOfValuesForCurrentDate);
        }

        private void FillDateCell(int firstValueIndex, int numberOfValuesForDate)
        {
            int startRow = 1 + firstValueIndex * ScheduleTable.RowsPerValue;
            ScheduleTableCell cell = ScheduleTable.Cell(startRow, 0);
            cell.RowSpan = numberOfValuesForDate * ScheduleTable.RowsPerValue;
            cell.CssClass = Options.RangeHeaderCss;
            cell.IsHeader = true;
            cell.Data = Convert.ToDateTime(ArrRangeValues[firstValueIndex]);
            cell.ItemType = ScheduleItemType.DateHeader;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// calculate the TitleIndex in the table, given the objTitleValue
        /// </summary>
        /// <param name="objTitleValue"></param>
        /// <returns></returns>
        /// -----------------------------------------------------------------------------
        protected override int CalculateTitleIndex(object objTitleValue)
        {
            // Find the title index by matching with the title values array
            for (int k = 0; k < ArrTitleValues.Count; k++)
            {
                if (ArrTitleValues[k].ToString() == objTitleValue.ToString())
                {
                    return k + 1;
                }
            }
            return -1;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Calculate the range cell index in the table, given the objRangeValue and the objTitleValue
        /// The result is the real row index in the table
        /// </summary>
        /// <param name="objRangeValue">The range value from the data source</param>
        /// <param name="objTitleValue">The title value from the data source</param>
        /// <param name="isEndValue">False if we're calculating the range value index for the start of the item, True if it's the end</param>
        /// <returns>The range cell index (row index)</returns>
        /// -----------------------------------------------------------------------------
        protected override int CalculateRangeCellIndex(object objRangeValue, object objTitleValue, bool isEndValue)
        {
            if (objRangeValue == null || object.ReferenceEquals(objRangeValue, DBNull.Value))
                return -1;
            int rangeValueIndex = -1;
            // Find range value index by matching with range values array
            if (BaseOptions.FullTimeScale && !(objRangeValue is DateTime))
            {
                throw new MvcScheduleException("The range field should be of type DateTime when FullTimeScale is set to true");
            }
            if (BaseOptions.FullTimeScale)
            {
                DateTime dObj = (DateTime)objRangeValue;
                // if no match is found, use the index of the EndOfTimeScale value
                rangeValueIndex = ArrRangeValues.Count;
                for (int k = 0; k < ArrRangeValues.Count; k++)
                {
                    DateTime Dk = (DateTime)ArrRangeValues[k];
                    if (dObj <= Dk)
                    {
                        if (k == 0 && isEndValue)
                        {
                            // This can happen when the end value is 24:00:00, which will
                            // match with the value 0:00:00 and give k=0
                            // Instead of the value k=0, use k=arrRangeValues.Count-1
                            rangeValueIndex = ArrRangeValues.Count;
                        }
                        else
                        {
                            rangeValueIndex = k + 1;
                        }
                        break; 
                    }
                }               
            }
            else // ! FullTimeScale
            {
                // find the matching value in arrRangeValues
                for (int k = 0; k < ArrRangeValues.Count; k++)
                {
                    if (ArrRangeValues[k].ToString() == objRangeValue.ToString())
                    {
                        rangeValueIndex = k + 1;
                        break; 
                    }
                }
            }
            if (!BaseOptions.IncludeEndValue && BaseOptions.ShowValueMarks)
            {
                // Each item spans two rows
                return rangeValueIndex * 2;
            }
            else
            {
                return rangeValueIndex;
            }
        }

        protected override int GetTitleCount()
        {
            return ArrTitleValues.Count;
        }

        protected override int GetRangeHeaderColumnIndex()
        {
            // when SeparateDateHeader=True, the first index (column or row) is the date header,
            // the second (column or row) contains the range values
            return Options.SeparateDateHeader ? 1 : 0;
        }

        protected override void AddTitleHeaderData()
        {
            int titleCount = GetTitleCount();

            // iterate arrTitleValues creating a new item for each data item
            for (int column = 1; column <= titleCount; column++)
            {
                ScheduleTableCell cell = ScheduleTable.Cell(0, column);
                cell.Data = ArrTitleValues[column - 1];
                cell.ItemType = ScheduleItemType.TitleHeader;
            }
        }

        protected override IComparer<ScheduleItemData<TItem>> GetComparer()
        {
            if (Options.AutoSortTitles)
            {
                return new ScheduleGeneralComparer<TItem>();
            }
            else
            {
                return null;
            }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Get a format when there's no template
        /// </summary>
        /// <param name="type">Type of the item</param>
        /// <returns>A format</returns>
        /// -----------------------------------------------------------------------------
        protected override string GetFormat(ScheduleItemType? type)
        {
            switch (type)
            {
                case ScheduleItemType.TitleHeader:
                    return Options.TitleDataFormatString;
                case ScheduleItemType.RangeHeader:
                    return Options.RangeDataFormatString;
                case ScheduleItemType.DateHeader:
                    return Options.DateHeaderDataFormatString;
            }
            return null;
        }

        protected override string GetStyle(ScheduleItemType type)
        {
            // handle DateHeader, which is not handled in the base class
            if (type == ScheduleItemType.DateHeader)
            {
                return Options.RangeHeaderCss;
            }
            return base.GetStyle(type);
        }

        ///' -----------------------------------------------------------------------------
        ///' <summary>
        ///' Calculate the title (data source value) given the cell index
        ///' </summary>
        ///' <param name="titleIndex">Title index of the cell</param>
        ///' <returns>Object containing the title</returns>
        ///' -----------------------------------------------------------------------------
        protected override object CalculateTitle(int titleIndex, int cellIndex)
        {
            return ArrTitleValues[titleIndex - 1];
        }

        protected override void RenderItem(ScheduleTableCell cell, HtmlHelper htmlHelper, HtmlTextWriter htw)
        {
            if (cell.ItemType == ScheduleItemType.DateHeader)
            {
                if (DateHeaderDisplayExpression != null)
                {
                    htw.Write(DateHeaderDisplayExpression.Compile().Invoke((DateTime)cell.Data));
                }
                else
                {
                    htw.Write(cell.Data.ToString());
                }
            }
            else
            {
                base.RenderItem(cell, htmlHelper, htw);
            }
        }

        #endregion

    }

}