﻿using MvcSchedule.Data;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Text;
using System.Threading.Tasks;

namespace MvcSchedule.Objects
{

	/// <summary>
	/// Comparer class for the ScheduleCalendar control.
	/// </summary>
    internal class ScheduleCalendarComparer<TItem> : IComparer<ScheduleItemData<TItem>>
	{

		private bool _timeExpressionsContainDate;
        private Expression<Func<TItem, object>> _dateExpression;
		private DateTime _startDate;
		private int _numberOfDays;

        public ScheduleCalendarComparer(bool timeExpressionsContainDate, Expression<Func<TItem, object>> dateExpression, DateTime startDate, int numberOfDays)
		{
			_timeExpressionsContainDate = timeExpressionsContainDate;
            _dateExpression = dateExpression;
			_startDate = startDate;
			_numberOfDays = numberOfDays;
		}

        public int Compare(ScheduleItemData<TItem> x, ScheduleItemData<TItem> y)
		{
			if (!_timeExpressionsContainDate) {
				DateTime xDateValue = (DateTime)_dateExpression.Compile().Invoke(x.DataObject);
                DateTime yDateValue = (DateTime)_dateExpression.Compile().Invoke(y.DataObject);
				if (xDateValue != yDateValue) {
					return this.CompareDate(xDateValue, yDateValue);
				}
			}
            // In Calendar Mode with TimeExpressionsContainDate=True, there is no DateExpression
			DateTime xStartTime = (DateTime)x.StartValue;
			DateTime yStartTime = (DateTime)y.StartValue;
			if (xStartTime == yStartTime) {
				return ((DateTime)x.EndValue).CompareTo((DateTime)y.EndValue);
			} else {
				if (xStartTime.Date == yStartTime.Date) {
					return xStartTime.CompareTo(yStartTime);
				} else {
					return CompareDate(xStartTime.Date, yStartTime.Date);
				}
			}
		}

		private int CompareDate(DateTime date1, DateTime date2)
		{
			// When we have multiple weeks, we want to sort by the day first, then by the week.
			// This is because we process by column (columns being days of the week).
			// When we have days 1 to 14, the sorted result should be:
			// 1, 8, 2, 9, 3, 10, 4, 11, 5, 12, 6, 13, 7, 14
			int day1 = (date1 - _startDate).Days % _numberOfDays;
			int day2 = (date2 - _startDate).Days % _numberOfDays;
			if (day1 == day2) {
				int week1 = (date1 - _startDate).Days / _numberOfDays;
				int week2 = (date2 - _startDate).Days / _numberOfDays;
				return week1 - week2;
			} else {
				return day1 - day2;
			}
		}

	}

}

