﻿using MvcSchedule.Data;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.ComponentModel;
using System.Linq.Expressions;

namespace MvcSchedule.Objects
{

    // -------------------------- Databound Schedule Control v1.0.0 -----------------------
    //
    // ------------------------------ www.rekenwonder.com ---------------------------------
    //
    // ----------- Licensing: These controls are free under the LGPL license. -------------
    // ----------- See: http://www.gnu.org/licenses/lgpl.html for more details. -----------

    /// -----------------------------------------------------------------------------
    /// Project	 : MvcSchedule
    /// Class	 : ScheduleCalendar
    ///
    /// -----------------------------------------------------------------------------
    /// <summary>
    /// The ScheduleCalendar control is designed to represent a schedule in a calendar format.
    /// </summary>
    /// -----------------------------------------------------------------------------
    internal class ScheduleCalendar<TItem> : BaseSchedule<TItem>
    {

        #region Private members

        private MvcScheduleCalendarOptions _options;

        #endregion

        #region Properties

        public MvcScheduleCalendarOptions Options
        {
            get { return _options; }
            set { _options = value; }
        }

        protected override MvcScheduleOptions BaseOptions
        {
            get { return Options; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Whether to show the EmptyDataTemplate or not when no data is found
        /// </summary>
        /// <remarks>
        /// Overrides default value (True)
        /// </remarks>
        /// -----------------------------------------------------------------------------
        protected override bool ShowEmptyDataTemplate
        {
            // When FullTimeScale=True, an empty calendar is shown, so there's no need for a replacement
            get { return !Options.FullTimeScale; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The expression giving the start time of the events. This expression should also contain the date when TimeExpressionsContainDate==true
        /// </summary>
        /// <remarks>
        /// StartTimeExpression replaces DataRangeStartExpression for ScheduleCalendar
        /// </remarks>
        /// -----------------------------------------------------------------------------
        [Description("The expression giving the start time of the events. This expression should also contain the date when TimeExpressionsContainDate==true")]
        [Category("Data")]
        public Expression<Func<TItem, object>> StartTimeExpression
        {
            get { return base.DataRangeStartExpression; }
            set { base.DataRangeStartExpression = value; }
        }

        // Hide DataRangeStartField. For ScheduleCalendar, it's called StartTimeField
        [Browsable(false)]
        [Obsolete("The DataRangeStartExpression property is obsolete")]
        public new Expression<Func<TItem, object>> DataRangeStartExpression
        {
            get { return base.DataRangeStartExpression; }
            set { base.DataRangeStartExpression = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The expression giving the end time of the events. This expression should also contain the date when TimeExpressionsContainDate=true
        /// </summary>
        /// <remarks>
        /// EndTimeExpression replaces DataRangeEndExpression for ScheduleCalendar
        /// </remarks>
        /// -----------------------------------------------------------------------------
        [Description("The expression giving the end time of the events. This expression should also contain the date when TimeExpressionsContainDate=true")]
        [Category("Data")]
        public Expression<Func<TItem, object>> EndTimeExpression
        {
            get { return base.DataRangeEndExpression; }
            set { base.DataRangeEndExpression = value; }
        }

        // Hide DataRangeEndField. For ScheduleCalendar, it's called EndTimeField
        [Browsable(false)]
        [Obsolete("The DataRangeEndField property is obsolete")]
        public new Expression<Func<TItem, object>> DataRangeEndExpression
        {
            get { return base.DataRangeEndExpression; }
            set { base.DataRangeEndExpression = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The expression providing the dates. Ignored when TimeExpressionsContainDate==true. When TimeExpressionsContainDate==false, this expression should yield a result of type Date
        /// </summary>
        /// <remarks>
        /// DateExpression replaces TitleExpression for ScheduleCalendar
        /// </remarks>
        /// -----------------------------------------------------------------------------
        [Description("The expression providing the dates. Ignored when TimeExpressionsContainDate==true. When TimeExpressionsContainDate==false, this expression should yield a result of type Date.")]
        [Category("Data")]
        public Expression<Func<TItem, object>> DateExpression
        {
            get { return base.TitleExpression; }
            set { base.TitleExpression = value; }
        }

        // Hide TitleField. For ScheduleCalendar, it's called DateField
        [Browsable(false)]
        [Obsolete("The TitleField property is obsolete")]
        public new Expression<Func<TItem, object>> TitleExpression
        {
            get { return base.TitleExpression; }
            set { base.TitleExpression = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The expression giving the display format for the times. 
        /// </summary>
        /// <remarks>
        /// TimeDisplayExpression replaces DataRangeDisplayExpression for ScheduleCalendar
        /// </remarks>
        /// -----------------------------------------------------------------------------
        [Description("The expression giving the display format for the times")]
        [Category("Data")]
        public Expression<Func<object, string>> TimeDisplayExpression
        {
            get { return base.DataRangeDisplayExpression; }
            set { base.DataRangeDisplayExpression = value; }
        }

        // Hide DataRangeDisplayExpression. For ScheduleCalendar, it's called TimeDisplayExpression
        [Browsable(false)]
        [Obsolete("The DataRangeDisplayExpression property is obsolete")]
        public new Expression<Func<object, string>> DataRangeDisplayExpression
        {
            get { return base.DataRangeDisplayExpression; }
            set { base.DataRangeDisplayExpression = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The expression giving the display format for the dates. 
        /// </summary>
        /// <remarks>
        /// DateDisplayExpression replaces TitleDisplayExpression for ScheduleCalendar
        /// </remarks>
        /// -----------------------------------------------------------------------------
        [Description("The expression giving the display format for the dates")]
        [Category("Data")]
        public Expression<Func<object, string>> DateDisplayExpression
        {
            get { return base.TitleDisplayExpression; }
            set { base.TitleDisplayExpression = value; }
        }

        // Hide TitleDisplayExpression. For ScheduleCalendar, it's called DateDisplayExpression
        [Browsable(false)]
        [Obsolete("The TitleDisplayExpression property is obsolete")]
        public new Expression<Func<object, string>> TitleDisplayExpression
        {
            get { return base.TitleDisplayExpression; }
            set { base.TitleDisplayExpression = value; }
        }

        #endregion

        #region Methods

        // Check if all properties are set to make the control work
        protected internal override void CheckConfiguration()
        {
            base.CheckConfiguration();
            if (!Options.TimeExpressionsContainDate && (DateExpression == null))
                throw new MvcScheduleException("Either the DateExpression must be set, or TimeExpressionsContainDate must be true");
            if (StartTimeExpression == null)
                throw new MvcScheduleException("The StartTimeExpression is not set");
            if (EndTimeExpression == null)
                throw new MvcScheduleException("The EndTimeExpression is not set");
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// create the list with all times (Start or End)
        /// </summary>
        /// <param name="dataList"></param>
        /// -----------------------------------------------------------------------------
        protected override void FillRangeValueArray(ScheduleItemDataList<TItem> dataList)
        {
            ArrRangeValues = new ArrayList();
            if (Options.FullTimeScale)
            {
                TimeSpan tsInc = new TimeSpan(0, Options.TimeScaleInterval, 0);
                // ignore data, just fill the time scale
                // add incrementing times to the array
                TimeSpan t = Options.StartOfTimeScale;
                while (TimeSpan.Compare(t, Options.EndOfTimeScale) < 0)
                {
                    // use DateTime objects for easy display
                    DateTime dt = new DateTime(1, 1, 1, t.Hours, t.Minutes, 0);
                    ArrRangeValues.Add(dt);
                    t = t.Add(tsInc);
                }
                // Add the end of the timescale as well to make sure it's there
                // e.g. in the case of EndOfTimeScale=23:59 and TimeScaleInterval=1440, this is imperative
                DateTime dtEnd = new DateTime(1, 1, 1 + Options.EndOfTimeScale.Days, Options.EndOfTimeScale.Hours, Options.EndOfTimeScale.Minutes, 0);
                ArrRangeValues.Add(dtEnd);
                
            }
            else // Not FullTimeScale
            {
                // Just add the times from the data source
                foreach (ScheduleItemData<TItem> item in dataList)
                {
                    object t1 = item.StartValue;
                    object t2 = item.EndValue;
                    if (!Options.TimeExpressionsContainDate)
                    {
                        ArrRangeValues.Add(t1);
                        ArrRangeValues.Add(t2);
                        
                    }
                    else // TimeExpressionsContainDate
                    {
                        // both t1 and t2 should be of type DateTime now
                        if (!(t1 is DateTime))
                        {
                            throw new MvcScheduleException("When TimeExpressionsContainDate is true, StartTimeExpression should yield a result of type DateTime");
                        }
                        DateTime dt1 = (DateTime)t1;
                        if (!(t2 is DateTime))
                        {
                            throw new MvcScheduleException("When TimeExpressionsContainDate is true, EndTimeExpression should yield a result of type DateTime");
                        }
                        DateTime dt2 = (DateTime)t2;
                        // remove date part, only store time part in array
                        ArrRangeValues.Add(new DateTime(1, 1, 1, dt1.Hour, dt1.Minute, dt1.Second));
                        if (dt2.Hour > 0 || dt2.Minute > 0 || dt2.Second > 0)
                        {
                            ArrRangeValues.Add(new DateTime(1, 1, 1, dt2.Hour, dt2.Minute, dt2.Second));
                        }
                        else
                        {
                            // if the end is 0:00:00 hour, insert 24:00:00 hour instead
                            ArrRangeValues.Add(new DateTime(1, 1, 2, 0, 0, 0));
                        }
                    }
                }
            }

            ArrRangeValues.Sort();
            ArrRangeValues.RemoveDoubles();
        }

        /// <summary>
        /// When TimeExpressionsContainDate==true, items could span over midnight, even several days.
        /// Split them.
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        /// -----------------------------------------------------------------------------
        public override ScheduleItemDataList<TItem> PreprocessData(IEnumerable<TItem> data)
        {
            ScheduleItemDataList<TItem> dataList = base.PreprocessData(data);
            ShiftStartDate();
            if (!Options.TimeExpressionsContainDate) return dataList;
            if (data == null) return dataList;
            for (int i = 0; i < dataList.Count; i++)
            {
                ScheduleItemData<TItem> currentItem = dataList[i];
                DateTime dtStartValue = (DateTime)currentItem.StartValue;
                DateTime dtEndValue = (DateTime)currentItem.EndValue;
                DateTime dateStart = dtStartValue.Date;
                DateTime dateEnd = dtEndValue.Date;
                if (dtEndValue.Hour == 0 && dtEndValue.Minute == 0 && dtEndValue.Second == 0)
                {
                    // when it ends at 0:00:00 hour, it's representing 24:00 hours of the previous day
                    dateEnd = dateEnd.AddDays(-1);
                }
                // Check if the item spans midnight. If so, split it.
                if (dateStart < dateEnd)
                {
                    // the item spans midnight. We'll truncate the item first, so that it only
                    // covers the last day, and we'll add new items for the other day(s) in the loop below.
                    if (Options.FullTimeScale)
                    {
                        // truncate the item by setting its start time to StartOfTimeScale
                        currentItem.StartValue = new DateTime(dateEnd.Year, dateEnd.Month, dateEnd.Day, Options.StartOfTimeScale.Hours, Options.StartOfTimeScale.Minutes, Options.StartOfTimeScale.Seconds);
                    }
                    else
                    {
                        // truncate the item by setting its start time to 0:00:00 hours
                        currentItem.StartValue = dateEnd.Date;
                    }
                }
                while (dateStart < dateEnd)
                {
                    // If the item spans midnight once, create an additional item for the first day.
                    // If it spans midnight several times, create additional items for each day.
                    var extraItem = new ScheduleItemData<TItem>(currentItem);
                    extraItem.StartValue = dtStartValue;
                    if (Options.FullTimeScale)
                    {
                        // set the end time to the EndOfTimeScale value
                        DateTime dateEnd2 = new DateTime(dateStart.Year, dateStart.Month, dateStart.Day, Options.EndOfTimeScale.Hours, Options.EndOfTimeScale.Minutes, Options.EndOfTimeScale.Seconds);
                        if (Options.EndOfTimeScale.Equals(new TimeSpan(1, 0, 0, 0)))
                        {
                            // EndOfTimeScale is 24:00 hours. Set the end at 0:00 AM of the next day.
                            // We'll catch this case later and show the proper value.
                            dateEnd2 = dateEnd2.AddDays(1);
                        }
                        extraItem.EndValue = dateEnd2;
                    }
                    else
                    {
                        // Set the end time to 24:00 hours. This is 0:00 AM of the next day.
                        // We'll catch this case later and show the proper value.
                        extraItem.EndValue = dateStart.Date.AddDays(1);
                    }
                    dataList.Add(extraItem);
                    dateStart = dateStart.AddDays(1);
                    if (Options.FullTimeScale)
                    {
                        // next item should start at StartOfTimeScale
                        dtStartValue = new DateTime(dateStart.Year, dateStart.Month, dateStart.Day, Options.StartOfTimeScale.Hours, Options.StartOfTimeScale.Minutes, Options.StartOfTimeScale.Seconds);
                    }
                    else
                    {
                        // next item should start at 0:00:00 hour
                        dtStartValue = dateStart.Date;
                    }
                }
            }
            return dataList;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Calculate the TitleIndex in the table, given the objTitleValue
        /// </summary>
        /// <param name="objTitleValue"></param>
        /// <returns></returns>
        /// -----------------------------------------------------------------------------
        protected override int CalculateTitleIndex(object objTitleValue)
        {
            if (!(objTitleValue is DateTime))
            {
                throw new MvcScheduleException("Date expression should yield a result of type DateTime in Calendar mode");
            }
            DateTime dtDate = Convert.ToDateTime(objTitleValue);            
            dtDate = dtDate.Date; // remove time part, if any
            return (((dtDate.Subtract(Options.StartDate.Date)).Days) % Options.NumberOfDays) + 1;  
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Calculate the row index in the table, given the objRangeValue and the objTitleValue
        /// The result is the real row index in the table
        /// </summary>
        /// <param name="objRangeValue">The range value from the data source</param>
        /// <param name="objTitleValue">The title value from the data source</param>
        /// <param name="isEndValue">False if we're calculating the row index for the start of the item, True if it's the end</param>
        /// <returns>The row index</returns>
        /// -----------------------------------------------------------------------------
        protected override int CalculateRangeCellIndex(object objRangeValue, object objTitleValue, bool isEndValue)
        {
            // Find row index by matching with range values array
            if (Options.FullTimeScale && !(objRangeValue is DateTime))
            {
                throw new MvcScheduleException("The time expression should yield a result of type DateTime when FullTimeScale is set to true");
            }
            if (Options.TimeExpressionsContainDate && !(objRangeValue is DateTime))
            {
                throw new MvcScheduleException("The time expression should should yield a result of type DateTime when TimeExpressionsContainDate is set to true");
            }
            int rowInWeek = CalculateRowInWeek(objRangeValue, isEndValue);
            if (rowInWeek == -1)
                return -1;

            if (!Options.IncludeEndValue && Options.ShowValueMarks)
            {
                // Each item spans two rows
                rowInWeek = rowInWeek * 2;
            }

            // The rowInWeek that we found corresponds with an item in the first week.
            // If the item is in another week, modify the value accordingly.
            // To find out, check the date of the item
            DateTime dtDate = GetDate(objRangeValue, objTitleValue, isEndValue);

            // if dtDate is more than NumberOfDays after StartDate, add additional rows
            int rowsPerWeek = 1 + ArrRangeValues.Count;
            if (!Options.IncludeEndValue && Options.ShowValueMarks)
                rowsPerWeek = 1 + ArrRangeValues.Count * 2;

            return rowInWeek + (dtDate.Subtract(Options.StartDate).Days / Options.NumberOfDays) * rowsPerWeek;
        }

        protected override int GetTitleCount()
        {
            return Options.NumberOfDays; // make a title cell for every NumberOfDays            
        }

        protected override void AddTitleHeaderData()
        {
            int titleCount = GetTitleCount();

            // iterate through ArrTitleValues creating a new item for each data item
            for (int column = 1; column <= titleCount; column++)
            {
                int iWeek = 0;
                for (iWeek = 0; iWeek <= Options.NumberOfRepetitions - 1; iWeek++)
                {
                    DateTime obj = Options.StartDate.AddDays(column - 1 + iWeek * Options.NumberOfDays);
                    int rowsPerWeek = 1 + ArrRangeValues.Count;
                    if (!Options.IncludeEndValue && Options.ShowValueMarks)
                        rowsPerWeek = 1 + ArrRangeValues.Count * 2;
                    int row = iWeek * rowsPerWeek;

                    ScheduleTableCell cell = ScheduleTable.Cell(row, column);
                    cell.Data = obj;
                    cell.ItemType = ScheduleItemType.TitleHeader;
                }
            }
        }

        protected override object GetTitleValue(ScheduleItemData<TItem> dataItem)
        {
            if (Options.TimeExpressionsContainDate)
            {
                // When TimeExpressionsContainDate==true, use StartValue as Title
                return dataItem.StartValue;
            }
            else
            {
                return dataItem.TitleValue;
            }
        }

        public void ShiftStartDate()
        {
            if (Options.NumberOfDays != 7)
                return;
            // change the start date only for a weekly calendar
            // for any StartDate set by the user, shift it to the previous day indicated by the StartDay property
            // (by default, this will be the previous Monday)
            if (Convert.ToInt32(Options.StartDay) > Convert.ToInt32(Options.StartDate.DayOfWeek))
            {
                Options.StartDate = Options.StartDate.AddDays(-7);
            }
            Options.StartDate = Options.StartDate.AddDays(Convert.ToInt32(Options.StartDay) - Convert.ToInt32(Options.StartDate.DayOfWeek));
            // StartDate should be on the day of the week indicated by StartDay now
        }

        protected override IComparer<ScheduleItemData<TItem>> GetComparer()
        {
            return new ScheduleCalendarComparer<TItem>(Options.TimeExpressionsContainDate, DateExpression, Options.StartDate, Options.NumberOfDays);
        }

        protected override int GetRepetitionCount()
        {
            return Options.NumberOfRepetitions;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Get a format when there's no template
        /// </summary>
        /// <param name="type">Type of the item</param>
        /// <returns>A format</returns>
        /// -----------------------------------------------------------------------------
        protected override string GetFormat(ScheduleItemType? type)
        {
            switch (type)
            {
                case ScheduleItemType.TitleHeader:
                    return Options.DateFormatString;
                case ScheduleItemType.RangeHeader:
                    return Options.TimeFormatString;
            }
            return null;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Calculate the title value given the cell index
        /// </summary>
        /// <param name="titleIndex">Title index of the cell</param>
        /// <returns>Object containing the title</returns>
        /// -----------------------------------------------------------------------------
        protected override object CalculateTitle(int titleIndex, int cellIndex)
        {
            int cellsPerWeek = 0;
            if (!Options.IncludeEndValue && Options.ShowValueMarks)
            {
                cellsPerWeek = ArrRangeValues.Count * 2 + 1;
            }
            else
            {
                cellsPerWeek = ArrRangeValues.Count + 1;
            }
            int week = cellIndex / cellsPerWeek;
            return Options.StartDate.AddDays(titleIndex - 1 + week * 7);
        }

        #endregion

        #region Private methods

        /// <summary>
        /// Calculate the row for the given value.
        /// The first row is 1 (because the title is on row 0).
        /// </summary>
        /// <param name="objRangeValue"></param>
        /// <param name="isEndValue"></param>
        /// <returns></returns>
        private int CalculateRowInWeek(object objRangeValue, bool isEndValue)
        {
            if (Options.FullTimeScale)
            {
                DateTime dObj = (DateTime)objRangeValue;                
                dObj = new DateTime(1, 1, 1, dObj.Hour, dObj.Minute, dObj.Second); // omit the date part for comparison
                for (int k = 0; k < ArrRangeValues.Count; k++)
                {
                    DateTime dk = (DateTime)ArrRangeValues[k];
                    dk = new DateTime(1, 1, 1, dk.Hour, dk.Minute, dk.Second);  // omit date part
                    if (dObj < dk && k == 0 && isEndValue)
                    {
                        // value sits before start of time scale
                        if (dObj.Hour == 0 && dObj.Minute == 0 && dObj.Second == 0)
                        {
                            // This can happen when the time scale doesn't start at 0:00 
                            // and when the end value is 24:00:00, which will give dObj < dk
                            // Instead of the value k=0, use k=arrRangeValues.Count-1 (end of time scale)
                            return ArrRangeValues.Count;
                        }
                        else
                        {
                            return -1;
                        }
                    }
                    if (dObj >= dk && k == ArrRangeValues.Count - 1 && !isEndValue)
                    {
                        // value sits at or after the end of the time scale
                        return -1;
                    }
                    if (dObj <= dk)
                    {
                        // we found the row whose value is equal to or just above our value
                        if (k == 0 && isEndValue)
                        {
                            // This can happen when the end value is 24:00:00, which will
                            // match with the value 0:00:00 and give k=0
                            // Instead of the value k=0, use k=arrRangeValues.Count-1
                            return ArrRangeValues.Count;
                        }
                        else
                        {
                            return k + 1;
                        }
                    }
                }
                // if no match is found, use the index of the EndOfTimeScale value
                return ArrRangeValues.Count;                
            }
            else // Not FullTimeScale
            {
                if (!Options.TimeExpressionsContainDate)
                {
                    // find the matching value in arrRangeValues
                    for (int k = 0; k < ArrRangeValues.Count; k++)
                    {
                        if (ArrRangeValues[k].ToString() == objRangeValue.ToString())
                        {
                            return k + 1;
                        }
                    }
                }
                else // TimeExpressionsContainDate=True
                {                    
                    DateTime dObj = (DateTime)objRangeValue;                    
                    dObj = new DateTime(1, 1, 1, dObj.Hour, dObj.Minute, dObj.Second); // omit the date part for comparison
                    for (int k = 0; k < ArrRangeValues.Count; k++)
                    {
                        DateTime dk = (DateTime)ArrRangeValues[k];
                        dk = new DateTime(1, 1, 1, dk.Hour, dk.Minute, dk.Second);  // omit date part
                        if (dObj == dk)
                        {
                            if (k == 0 && isEndValue)
                            {
                                // This can happen when the end value is 24:00:00, which will
                                // match with the value 0:00:00, and give k=0
                                // Instead of the value k=0, use k=arrRangeValues.Count-1
                                return ArrRangeValues.Count;
                            }
                            else
                            {
                                return k + 1;
                            }
                        }
                    }
                }
                return -1;  // should never happen
            }
        }

        private DateTime GetDate(object objRangeValue, object objTitleValue, bool isEndValue)
        {
            if (!Options.TimeExpressionsContainDate)
            {
                // use objTitleValue for the date
                if (!(objTitleValue is DateTime))
                {
                    throw new MvcScheduleException("The date expression should be of type DateTime in Calendar mode when TimeExpressionsContainDate==false.");
                }
                return Convert.ToDateTime(objTitleValue);
            }
            else
            {
                // use objRangeValue for the date
                DateTime dObj = (DateTime)objRangeValue;
                DateTime dtDate = new DateTime(dObj.Year, dObj.Month, dObj.Day);
                if (isEndValue && dObj.Hour == 0 && dObj.Minute == 0 && dObj.Second == 0)
                {
                    // when it's the end of the item and the time = 0:00 hours,
                    // it's representing 24:00 hours of the previous day
                    dtDate = dtDate.AddDays(-1);
                }
                return dtDate;
            }
        }

        #endregion

    }

}