﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MvcSchedule.Objects
{
    public class MvcScheduleOptions
    {

        private LayoutEnum _layout = LayoutEnum.Vertical;
        private bool _fullTimeScale = false;
        private int _timeScaleInterval = 60;
        private TimeSpan? _startOfTimeScale;
        private TimeSpan? _endOfTimeScale;
        private bool _showValueMarks = false;
        private bool _includeEndValue = false;
        private bool _spanItemsWhenPossible = true;
        private bool _gridlines = true;

        private string _generalCss;
        private string _itemCss;
        private string _alternatingItemCss;
        private string _rangeHeaderCss;
        private string _titleCss;
        private string _backgroundCss;

        /// <summary>
        /// The direction in which the item ranges are shown.
        /// </summary>
        public LayoutEnum Layout
        {
            get { return _layout; }
            set { _layout = value; }
        }
        
        /// <summary>
        /// Boolean value indicating if a full time scale should be shown.
        /// If true, show a full time scale.
        /// If false, show only the occurring values in the data source.
        /// </summary>
        /// <remarks>Default: false</remarks>
        public bool FullTimeScale
        {
            get { return _fullTimeScale; }
            set { _fullTimeScale = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The number of minutes between each mark on the time scale. Only used when FullTimeScale is true.
        /// </summary>
        /// -----------------------------------------------------------------------------
        [Description("The number of minutes between each mark on the time scale. Only used when FullTimeScale is true.")]
        [DefaultValue(60)]
        [Category("Behavior")]
        public int TimeScaleInterval
        {
            get
            {
                return _timeScaleInterval;
            }
            set
            {
                if (value == 0)
                {
                    throw new MvcScheduleException("TimeScaleInterval can not be 0");
                }
                _timeScaleInterval = value;
            }
        }

        /// <summary>
        /// The start of the time scale. Only used when FullTimeScale is true.
        /// </summary>
        public TimeSpan StartOfTimeScale
        {
            get { 
                if(_startOfTimeScale.HasValue)
                    return _startOfTimeScale.Value;
                if (FullTimeScale)
                    return new TimeSpan(0, 0, 0);
                // = 24:00
                return new TimeSpan(8, 0, 0);
            }
            set
            {
                if (value.Days == 0)
                {
                    _startOfTimeScale = value;
                }
                else
                {
                    throw new MvcScheduleException("StartOfTimeScale should be between 0:00:00 and 23:59:00");
                }
            }
        }

        /// <summary>
        /// The end of the time scale. Only used when FullTimeScale is true.
        /// </summary>
        public TimeSpan EndOfTimeScale
        {
            get { 
                if(_endOfTimeScale.HasValue)
                    return _endOfTimeScale.Value;
                if (FullTimeScale)
                    return new TimeSpan(1, 0, 0, 0);
                // = 24:00
                return new TimeSpan(17, 0, 0);
            }
            set {
                if (value.Days == 0 || value.Equals(new TimeSpan(1, 0, 0, 0)))
                {
                    _endOfTimeScale = value;
                }
                else
                {
                    throw new MvcScheduleException("EndOfTimeScale should be between 0:00:00 and 1.00:00:00 (24h)");
                }
            }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// If ShowValueMarks is true, value marks will be shown in the range header column or row.
        /// Applied only when IncludeEndValue is false.
        /// </summary>
        /// <remarks>Default: false</remarks>
        public bool ShowValueMarks
        {
            get { return _showValueMarks; }
            set { _showValueMarks = value; }
        }

        /// <summary>
        /// If IncludeEndValue is true, the event is shown including the end row or column
        /// </summary>
        /// <remarks>Default: false</remarks>
        public bool IncludeEndValue
        {
            get { return _includeEndValue; }
            set { _includeEndValue = value; }
        }

        /// <summary>
        /// If true, spans item's width/height to fill current column/row available empty space in vertical/horizontal layout.
        /// </summary>
        /// <remarks>Default: true</remarks>
        public bool SpanItemsWhenPossible
        {
            get { return _spanItemsWhenPossible; }
            set { _spanItemsWhenPossible = value; }
        }

        /// <summary>
        /// Boolean value indicating if gridlines should be shown in the schedule's tabular display.
        /// </summary>
        /// <remarks>Default: true</remarks>
        public bool Gridlines
        {
            get { return _gridlines; }
            set { _gridlines = value; }
        }

        #region "Styles"

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The style class applied to the entire schedule table.
        /// </summary>
        /// -----------------------------------------------------------------------------
        [Description("The style class applied to the entire schedule table. ")]
        [Category("Style")]
        public virtual string GeneralCss
        {
            get { return _generalCss; }
            set { _generalCss = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The style class applied to schedule items.
        /// </summary>
        /// -----------------------------------------------------------------------------
        [Description("The style class applied to schedule items. ")]
        [Category("Style")]
        public virtual string ItemCss
        {
            get { return _itemCss; }
            set { _itemCss = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The style class applied to alternating schedule items.
        /// </summary>
        /// -----------------------------------------------------------------------------
        [Description("The style class applied to alternating schedule items. ")]
        [Category("Style")]
        public virtual string AlternatingItemCss
        {
            get { return _alternatingItemCss; }
            set { _alternatingItemCss = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The style class applied to range header items.
        /// </summary>
        /// -----------------------------------------------------------------------------
        [Description("The style class applied to range header items. ")]
        [Category("Style")]
        public virtual string RangeHeaderCss
        {
            get { return _rangeHeaderCss; }
            set { _rangeHeaderCss = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The style class applied to title items.
        /// </summary>
        /// -----------------------------------------------------------------------------
        [Description("The style class applied to title items. ")]
        [Category("Style")]
        public virtual string TitleCss
        {
            get { return _titleCss; }
            set { _titleCss = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// The style class applied to the background cells.
        /// </summary>
        /// -----------------------------------------------------------------------------
        [Description("The style class applied to the background cells. ")]
        [Category("Style")]
        public virtual string BackgroundCss
        {
            get { return _backgroundCss; }
            set { _backgroundCss = value; }
        }

        #endregion

    }
}
