﻿using MvcSchedule.Data;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.Collections.Specialized;
using System.ComponentModel;
using System.IO; 
using System.Linq;
using System.Linq.Expressions;
using System.Web;
using System.Text;
using System.Web.Mvc;
using System.Web.Mvc.Html;
using System.Web.UI;

// -------------------------- MvcSchedule Controls v1.0.0.0 --------------------
//
// ------------------------------ www.rekenwonder.com ---------------------------------

// --- Licensing: These controls are free and open source under the LGPL license. -----
// ----------- See: http://www.gnu.org/licenses/lgpl.html for more details. -----------

// KNOWN BUGS
// 1) When some templates are missing and you use the control in a form with postback.
//    If a template is missing, the control will provide default content (it will just show
//    the contents of the database field).
//    On postback however, this default content gets lost, and the table headers will stay empty.
//    For now, the only work-around is to provide all the templates if you need postback.
// 2) Events that span midnight in calendar mode should not get alternating colors
// 3) When FullTimeScale=True, the control may behave strangely when events end after the
//    EndOfTimeScale value. Several events may be merged into one cell, etc.
//    Therefore, make sure that events always end inside the time scale when FullTimeScale=True

// Any help with these problems will be mostly appreciated.

/// -----------------------------------------------------------------------------
/// Project	 : MvcSchedule
/// Class	 : BaseSchedule
///
/// -----------------------------------------------------------------------------
/// <summary>
/// BaseSchedule is the base class for the ScheduleGeneral and ScheduleCalendar controls
/// </summary>
/// -----------------------------------------------------------------------------
/// 
namespace MvcSchedule.Objects
{

    internal abstract class BaseSchedule<TItem> 
    {

        #region Private members

        private Expression<Func<TItem, object>> _dataRangeStartExpression;
        private Expression<Func<TItem, object>> _dataRangeEndExpression;
        private Expression<Func<TItem, object>> _titleExpression;
        private Expression<Func<TItem, string>> _itemDisplayExpression;
        private Expression<Func<object, string>> _dataRangeDisplayExpression;
        private Expression<Func<object, string>> _titleDisplayExpression;
        private string _itemPartialViewName;
        private string _emptyDataText = "No data to display";

        protected ScheduleTable ScheduleTable;

        private ArrayList _arrRangeValues;

        #endregion

        #region Public properties

        protected virtual MvcScheduleOptions BaseOptions
        {
            get;
            set;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// list with values to be shown in row header
        /// </summary>
        /// -----------------------------------------------------------------------------
        protected ArrayList ArrRangeValues
        {
            get { return _arrRangeValues; }
            set { _arrRangeValues = value; }
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Whether to show the EmptyDataTemplate or not when no data is found
        /// </summary>
        /// <remarks>
        /// Overridden in ScheduleCalendar when FullTimeScale=True
        /// </remarks>
        /// -----------------------------------------------------------------------------
        protected virtual bool ShowEmptyDataTemplate
        {
            // Overridden in ScheduleCalendar 
            get { return true; }
        }

        /// <summary>
        /// The expression that yields the start of a data item.
        /// </summary>
        public Expression<Func<TItem, object>> DataRangeStartExpression
        {
            get { return _dataRangeStartExpression; }
            set { _dataRangeStartExpression = value; }
        }

        /// <summary>
        /// The expression that yields the end of a data item.
        /// </summary>
        public Expression<Func<TItem, object>> DataRangeEndExpression
        {
            get { return _dataRangeEndExpression; }
            set { _dataRangeEndExpression = value; }
        }

        /// <summary>
        /// The expression that yields the title of a data item.
        /// </summary>
        public Expression<Func<TItem, object>> TitleExpression
        {
            get { return _titleExpression; }
            set { _titleExpression = value; }
        }

        /// <summary>
        /// The expression that yields the text to display in an item on the schedule.
        /// </summary>
        /// <remarks>When null, ItemPartialViewName must be provided.</remarks>
        public Expression<Func<TItem, string>> ItemDisplayExpression
        {
            get { return _itemDisplayExpression; }
            set { _itemDisplayExpression = value; }
        }

        /// <summary>
        /// The expression that yields the way to display the text of the start or end of a data item in the schedule.
        /// </summary>
        public Expression<Func<object, string>> DataRangeDisplayExpression
        {
            get { return _dataRangeDisplayExpression; }
            set { _dataRangeDisplayExpression = value; }
        }

        /// <summary>
        /// The expression that yields the text to display as the title of a data item in the schedule.
        /// </summary>
        public Expression<Func<object, string>> TitleDisplayExpression
        {
            get { return _titleDisplayExpression; }
            set { _titleDisplayExpression = value; }
        }

        /// <summary>
        /// The name of the partial view that displays an item on the schedule.
        /// </summary>
        /// <remarks>When null, ItemDisplayExpression must be provided.</remarks>
        public string ItemPartialViewName
        {
            get { return _itemPartialViewName; }
            set { _itemPartialViewName = value; }
        }

        /// <summary>
        /// Text shown when there's no data.
        /// For MvcScheduleCalendars with FullTimeScale set to true, the EmptyDataText is never shown.
        /// </summary>
        public string EmptyDataText
        {
            get { return _emptyDataText; }
            set { _emptyDataText = value; }
        }        

        #endregion

        #region Constructors

        // hide constructor in this abstract class
        protected BaseSchedule()
        {
        }

        #endregion

        #region Methods and Implementation

        /// <summary>
        /// Creates the HTML output used to render the Schedule control with the 
        /// specified data source.
        /// </summary>
        /// <param name="data">An IEnumerable that represents the data used to create 
        /// the control.</param>
        /// <param name="htmlHelper">Html Helper</param>
        public string RenderTable(IEnumerable<TItem> data, HtmlHelper htmlHelper)
        {
            if (data != null)
            {
                CheckConfiguration();

                // wrap the data in a ScheduleItemDataList
                ScheduleItemDataList<TItem> dataList = PreprocessData(data);

                if ((data == null || dataList.Count == 0) && ShowEmptyDataTemplate)
                {
                    return EmptyDataText;
                }
                else
                {
                    FillRangeValueArray(dataList);
                    FillTitleValueArray(dataList);

                    CreateEmptyTable();

                    AddRangeHeaderData();
                    AddTitleHeaderData();
                    AddData(dataList);
                    ScheduleTable.AddRangeValueMarks(BaseOptions.TitleCss, BaseOptions.RangeHeaderCss);
                    AddDateHeaderData();

                    return ConvertModelToTable(htmlHelper);
                }
            }
            else return string.Empty;
        }

        /// <summary>
        /// Check if all properties are set to make the control work
        /// Override this function in derived versions.
        /// </summary>
        protected internal virtual void CheckConfiguration()
        {
            if (ItemDisplayExpression == null && string.IsNullOrEmpty(ItemPartialViewName))
                throw new MvcScheduleException("Either ItemDisplayExpression or ItemPartialViewName must be provided");
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Create the list with all range header values (Start or End)
        /// </summary>
        /// <param name="dataList"></param>
        /// -----------------------------------------------------------------------------
        protected abstract void FillRangeValueArray(ScheduleItemDataList<TItem> dataList);

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Pre-process the data by transforming it into a list that can be easily manipulated
        /// Currently overridden in ScheduleCalendar to split the items that span midnight
        /// </summary>
        /// <param name="data">The enumerable object containing the data</param>
        /// -----------------------------------------------------------------------------
        public virtual ScheduleItemDataList<TItem> PreprocessData(IEnumerable<TItem> data)
        {
            // override to add useful processing if any
            var dataList = new ScheduleItemDataList<TItem>(this._dataRangeStartExpression, this._dataRangeEndExpression, this._titleExpression);
            if (data != null)
            {
                foreach (TItem item in data)
                {
                    dataList.AddItem(item);
                }
            }
            return dataList;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Create the list with all the title values
        /// </summary>
        /// <param name="dataList">The list object containing the data</param>
        /// -----------------------------------------------------------------------------
        protected virtual void FillTitleValueArray(ScheduleItemDataList<TItem> dataList)
        {
            // Override in ScheduleGeneral to fill title array
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Get the number of times the control should repeat the schedule
        /// (every time with new headers)
        /// </summary>
        /// <returns>An integer value indicating the repetition count</returns>
        /// <remarks>
        /// Overridden in ScheduleCalendar to return NumberOfRepetitions (usually number of weeks)
        /// </remarks>
        /// -----------------------------------------------------------------------------
        protected virtual int GetRepetitionCount()
        {
            return 1;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Create a table with the right number of rows and columns.
        /// The table is always built in vertical layout. 
        /// When building the actual html table later, the ScheduleTable is converted to the real layout.
        /// The actual content is added later.
        /// </summary>
        /// -----------------------------------------------------------------------------
        private void CreateEmptyTable()
        {
            int repetitionCount = GetRepetitionCount();
            if (!BaseOptions.IncludeEndValue && BaseOptions.ShowValueMarks)
            {
                ScheduleTable = new ScheduleTableWithValueMarks(repetitionCount);
            }
            else
            {
                ScheduleTable = new ScheduleTable(repetitionCount);
            }

            int columnCount = GetTitleCount();

            // one extra column for the range headers
            for (int column = 1; column <= columnCount + 1; column++)
            {
                ScheduleTable.AddColumn();
            }

            int rowCount = ArrRangeValues.Count;
            ScheduleTable.Initialize(rowCount, BaseOptions.TitleCss, BaseOptions.RangeHeaderCss, BaseOptions.BackgroundCss);
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Add date headers to the table when SeparateDateHeader=True
        /// Overridden only in ScheduleGeneral to add date header data
        /// </summary>
        /// -----------------------------------------------------------------------------
        protected virtual void AddDateHeaderData()
        {
            // Override in ScheduleGeneral to add date header data
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Iterate the arrRangeValues array, creating a new header item for each data item
        /// </summary>
        /// -----------------------------------------------------------------------------
        private void AddRangeHeaderData()
        {
            int cellsPerWeek = ScheduleTable.RowsPerRepetition(ArrRangeValues.Count);

            for (int week = 0; week < GetRepetitionCount(); week++)
            {
                for (int i = 0; i < ArrRangeValues.Count; i++)
                {
                    int row = i * ScheduleTable.RowsPerValue + 1 + week * cellsPerWeek;
                    ScheduleTableCell cell = ScheduleTable.Cell(row, 0);
                    cell.Data = ArrRangeValues[i];
                    cell.ItemType = ScheduleItemType.RangeHeader;
                }
            }
        }

        protected abstract void AddTitleHeaderData();

        protected abstract IComparer<ScheduleItemData<TItem>> GetComparer();

        /// <summary>
        /// Add the actual items from the data source to the body of the table
        /// </summary>
        /// <param name="dataList">The list that contains the data items</param>
        private void AddData(ScheduleItemDataList<TItem> dataList)
        {
            if (dataList == null) return;

            IComparer<ScheduleItemData<TItem>> comparer = this.GetComparer();
            if (comparer != null)
            {
                dataList.Sort(comparer);
            }

            // iterate through the data creating a new item for each data item
            int prevColumn = -1;
            var startRowsInThisColumn = new List<int>(); // list of start rows for each subcolumn

            //    make sure the data is processed in the right order: from bottom right up to top left.
            for (int i = dataList.Count - 1; i >= 0; i--)
            {
                ScheduleItemData<TItem> dataItem = dataList[i]; // this data item will be one entry in the schedule

                // Let's find out where it should be displayed
                // Even if the control is in Horizontal layout, we'll build the ScheduleTable in Vertical layout for simplicity, 
                // and flip it later
                object objTitleField = GetTitleValue(dataItem);
                int column = CalculateTitleIndex(objTitleField);
                if (column < 1) break; // since column is descending, and this one is too low already, skip all the rest too

                if (column != prevColumn)
                {
                    SpanSubcolumnsWhenPossible(prevColumn, startRowsInThisColumn.Count);
                    startRowsInThisColumn.Clear();
                }

                object objStartValue = dataItem.StartValue;
                object objEndValue = dataItem.EndValue;
                int startRow = CalculateRangeCellIndex(objStartValue, objTitleField, false);
                int endRow = CalculateRangeCellIndex(objEndValue, objTitleField, true);
                
                if (startRow > -1 && endRow > -1) // if not out of range
                {
                    if (!BaseOptions.IncludeEndValue)
                        endRow = endRow - 1;
                    int rowSpan = endRow - startRow + 1;
                    if (rowSpan == 0)
                        rowSpan = 1;

                    int maxStartRow = ScheduleTable.RowCount - 1;
                    if (startRow > 0 && startRow <= maxStartRow)
                    {
                        int subColumnOffset = -1;
                        if (column == prevColumn)
                        {
                            // look if there's another subcolumn where the time slot is still free
                            for (int columnOffset = 0; columnOffset < startRowsInThisColumn.Count; columnOffset++)
                            {
                                int startRowForSubcolumn = startRowsInThisColumn[columnOffset];
                                if (startRow + rowSpan <= startRowForSubcolumn)
                                {
                                    // found free time slot
                                    subColumnOffset = columnOffset;
                                    break; 
                                }
                            }

                            // no subcolumn found with free time slot
                            if (subColumnOffset == -1)
                            {
                                SplitColumn(column); // create a new subcolumn (to the left of the existing ones)                                
                                startRowsInThisColumn.Insert(0, startRow); // keep the value of its startRow                                
                                subColumnOffset = 0;
                            }
                            else
                            {
                                startRowsInThisColumn[subColumnOffset] = startRow;
                            }
                        }
                        else
                        {
                            subColumnOffset = 0;
                            startRowsInThisColumn.Add(startRow); // first item in this column: create an entry for the startRow                            
                        }

                        ScheduleTableCell cell = ScheduleTable.Cell(startRow, column + subColumnOffset);
                        // create new content
                        cell.Data = dataItem.DataObject;
                        if (i % 2 == 0)
                        {
                            cell.ItemType = ScheduleItemType.Item;
                            cell.CssClass = BaseOptions.ItemCss;
                        }
                        else
                        {
                            cell.ItemType = ScheduleItemType.AlternatingItem;
                            cell.CssClass = BaseOptions.AlternatingItemCss;
                        }

                        MergeCells(startRow, column + subColumnOffset, rowSpan);
                        // save location for next item to compare with
                        prevColumn = column;
                    }
                }
            }
            SpanSubcolumnsWhenPossible(prevColumn, startRowsInThisColumn.Count);
        }

        /// <summary>
        /// Given a column that's been split, span the cells over all its subcolumns where possible
        /// </summary>
        /// <param name="column">Index of column that has been split into subcolumns with the same title</param>
        /// <param name="subColumnCount">Number of columns with the same title</param>
        private void SpanSubcolumnsWhenPossible(int column, int subColumnCount)
        {
            if (column < 0) return;
            if (subColumnCount <= 1) return;
            int row = 0;
            while (row < ScheduleTable.RowCount)
            {
                ScheduleTableCell foundCell = null;
                int foundSubcolumnOffset = -1;
                for (int subColumnOffset = 0; subColumnOffset < subColumnCount; subColumnOffset++)
                {
                    ScheduleTableCell cell = ScheduleTable.Cell(row, column + subColumnOffset);
                    if (cell.ItemType.HasValue)
                    {
                        foundCell = cell;
                        foundSubcolumnOffset = subColumnOffset;
                        break; 
                    }
                }
                if (foundCell == null)
                {
                    // no items found, span the empty cells over all the subcolumns
                    ScheduleTable.Cell(row, column).ColumnSpan = subColumnCount;
                    row = row + 1;
                }
                else
                {
                    // check if the found item overlaps in the rows below
                    int currentRow = row;
                    int subColumnOffsetOfOverlappingCell = -1;
                    int rowspan = foundCell.RowSpan;
                    if (rowspan == 0)
                        rowspan = 1;
                    int nextRowThatCouldBeSpanned = row + rowspan;
                    // find the next row that could be spanned
                    // (the row not yet overlapped by items in the current row or overlapping items of those items)
                    // in the mean time, check if the current foundCell is not overlapping anything
                    while (row < nextRowThatCouldBeSpanned)
                    {
                        for (int subColumnOffset2 = 0; subColumnOffset2 < subColumnCount; subColumnOffset2++)
                        {
                            ScheduleTableCell cell = ScheduleTable.Cell(row, column + subColumnOffset2);
                            if (cell.ItemType == ScheduleItemType.Item || cell.ItemType == ScheduleItemType.AlternatingItem)
                            {
                                if (!cell.Equals(foundCell))
                                {
                                    // found an item in the same column that overlaps with the foundCell 
                                    // => foundCell can not be spanned
                                    if (subColumnOffsetOfOverlappingCell < 0)
                                    {
                                        subColumnOffsetOfOverlappingCell = subColumnOffset2;
                                    }
                                }
                                int endRow = row + cell.RowSpan - 1;
                                if (endRow + 1 > nextRowThatCouldBeSpanned)
                                {
                                    nextRowThatCouldBeSpanned = endRow + 1;
                                }
                            }
                        }
                        row += 1;
                    }
                    if (subColumnOffsetOfOverlappingCell < 0)
                    {
                        // no overlapping cells found
                        // span the cell over all the subcolumns
                        if (foundSubcolumnOffset > 0)
                        {
                            ScheduleTable.SwapCells(currentRow, column, column + foundSubcolumnOffset);
                        }
                        ScheduleTable.Cell(currentRow, column).ColumnSpan = subColumnCount;
                    }
                    if (subColumnOffsetOfOverlappingCell > foundSubcolumnOffset)
                    {
                        // span the cell over the subcolumns where there are no overlapping cells
                        if (foundSubcolumnOffset > 0)
                        {
                            ScheduleTable.SwapCells(currentRow, column, column + foundSubcolumnOffset);
                        }
                        ScheduleTable.Cell(currentRow, column).ColumnSpan = subColumnOffsetOfOverlappingCell;
                    }
                }
            }
        }

        protected virtual object GetTitleValue(ScheduleItemData<TItem> dataItem)
        {
            // overridden in ScheduleCalendar
            return dataItem.TitleValue;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Calculate the range cell index in the table, given the objRangeValue and the objTitleValue
        /// The result is the real row index in the table
        /// </summary>
        /// <param name="objRangeValue">The range value from the data source</param>
        /// <param name="objTitleValue">The title value from the data source</param>
        /// <param name="isEndValue">False if we're calculating the range value index for the 
        /// start of the item, True if it's the end</param>
        /// <returns>The range cell index</returns>
        /// -----------------------------------------------------------------------------
        protected abstract int CalculateRangeCellIndex(object objRangeValue, object objTitleValue, bool isEndValue);

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Calculate the TitleIndex in the table, given the objTitleValue
        /// </summary>
        /// <param name="objTitleValue">The title value from the data source</param>
        /// <returns>The title index</returns>
        /// -----------------------------------------------------------------------------
        protected abstract int CalculateTitleIndex(object objTitleValue);

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Merge cells starting at startRow
        /// </summary>
        /// <param name="startRow">Row of the first cell to merge.</param>
        /// <param name="column">Column containing the adjacent cells.</param>
        /// <param name="rowSpan">Number of rows that the newly merged cell should span.</param>
        /// -----------------------------------------------------------------------------
        private void MergeCells(int startRow, int column, int rowSpan)
        {
            if (column > ScheduleTable.ColumnCount - 1) return;
            if (startRow < 1) return;
            if (rowSpan < 2)  return;

            int maxRow = ScheduleTable.RowCount - 1;

            if (startRow > maxRow) return;

            if (startRow + rowSpan - 1 > maxRow)
            {
                rowSpan = maxRow - startRow + 1;
            }

            if (rowSpan <= ScheduleTable.Cell(startRow, column).RowSpan) return;

            // change span property to extend the cell
            // no actual merging is done
            // later in the method HideSpannedCells, the cells covered by the spanning are marked, so that they will not be rendered in html.
            ScheduleTable.Cell(startRow, column).RowSpan = rowSpan;
        }

        /// <summary>
        /// Split a column in 2 in order to allow the display of overlapping items.
        /// </summary>
        /// <param name="column">Column of the cells to be split</param>
        private void SplitColumn(int column)
        {
            // To show overlapping items, we simply split the entire column in 2,
            // This works well only when there are not yet any split columns to the left
            // If we work from right to left and bottom to top, this should be OK
            //
            // We'll split the column by inserting a whole new column 
            ScheduleTable.InsertColumn(column, BaseOptions.BackgroundCss);
        }

        protected virtual string GetStyle(ScheduleItemType type)
        {
            // Override to add additional styles (such as the style for DateHeaders in ScheduleGeneral)
            switch (type)
            {
                case ScheduleItemType.RangeHeader:
                    return BaseOptions.RangeHeaderCss;
                case ScheduleItemType.TitleHeader:
                    return BaseOptions.TitleCss;
                case ScheduleItemType.Item:
                    return BaseOptions.ItemCss;
                case ScheduleItemType.AlternatingItem:
                    if (BaseOptions.AlternatingItemCss != null)
                        return BaseOptions.AlternatingItemCss;
                    else
                        return BaseOptions.ItemCss;
            }
            return null;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// when there's no template, try to present the data in a reasonable format
        /// </summary>
        /// <param name="value">Value of the item</param>
        /// <param name="type">Type of the item</param>
        /// <returns>A formatted string</returns>
        /// -----------------------------------------------------------------------------
        protected string FormatDataValue(object value, ScheduleItemType? type)
        {
            if (value == null) return string.Empty;

            string formatString = GetFormat(type);
            if (!string.IsNullOrEmpty(formatString))
                return string.Format(formatString, value);
            else
                return value.ToString();
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Get a format when there's no template
        /// </summary>
        /// <param name="type">Type of the item</param>
        /// <returns>A format</returns>
        /// -----------------------------------------------------------------------------
        protected abstract string GetFormat(ScheduleItemType? type);

        protected abstract int GetTitleCount();

        protected virtual int GetRangeHeaderColumnIndex()
        {
            // overridden in ScheduleGeneral when using date headers
            return 0;
        }

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Calculate the title value given the cell index
        /// </summary>
        /// <param name="titleIndex">Title index of the cell</param>
        /// <returns>Object containing the title</returns>
        /// -----------------------------------------------------------------------------
        protected abstract object CalculateTitle(int titleIndex, int cellIndex);

        /// -----------------------------------------------------------------------------
        /// <summary>
        /// Calculate the range value index given the cell index
        /// </summary>
        /// <param name="row">Row of the cell</param>
        /// <param name="column">Column of the cell</param>
        /// <returns>Integer containing the range value index</returns>
        /// -----------------------------------------------------------------------------
        private int CalculateRangeValueIndex(int row, int column)
        {
            int rowsPerValue = 1;
            if (!BaseOptions.IncludeEndValue && BaseOptions.ShowValueMarks)
                rowsPerValue = 2;
            int rowsPerWeek = ArrRangeValues.Count * rowsPerValue + 1;
            int rangeValueIndex = (row % rowsPerWeek) / rowsPerValue - 1;
            if (rangeValueIndex < 0)
                rangeValueIndex = 0;
            return rangeValueIndex;
        }

        /// <summary>
        /// The table is first built in memory for a Vertical layout.
        /// This table is converted to the actual html table here in either layout (Vertical or Horizontal).
        /// </summary>
        /// <returns>The result html</returns>
        private string ConvertModelToTable(HtmlHelper htmlHelper)
        {
            ScheduleTable.HideSpannedCells();

            var sw = new StringWriter();
            var htw = new HtmlTextWriter(sw);
            WriteTableTag(htw);
            htw.Indent++;
            if (BaseOptions.Layout == LayoutEnum.Vertical)
            {
                for (int row = 0; row < ScheduleTable.RowCount; row++)
                {
                    htw.WriteLine();
                    htw.WriteFullBeginTag("tr");
                    htw.Indent++;
                    for (int column = 0; column < ScheduleTable.ColumnCount; column++)
                    {
                        ScheduleTableCell cell = ScheduleTable.Cell(row, column);
                        if (cell.Visible)
                            WriteCellVertical(htmlHelper, htw, cell);
                    }
                    htw.WriteLine();
                    htw.Indent--;
                    htw.WriteEndTag("tr");
                }                
            }
            else // Horizontal
            {
                // flip the ScheduleTable's columns/rows to get the html table
                for (int row = 0; row < ScheduleTable.ColumnCount; row++)
                {
                    htw.WriteLine();
                    htw.WriteFullBeginTag("tr");
                    htw.Indent++;
                    for (int column = 0; column < ScheduleTable.RowCount; column++)
                    {
                        ScheduleTableCell cell = ScheduleTable.Cell(column, row);
                        if (cell.Visible)
                            WriteCellHorizontal(htmlHelper, htw, cell);
                    }
                    htw.WriteLine();
                    htw.Indent--;
                    htw.WriteEndTag("tr");
                }
            }
            htw.WriteLine();
            htw.Indent--;
            htw.WriteEndTag("table");
            return sw.ToString();
        }

        private void WriteTableTag(HtmlTextWriter htw)
        {
            htw.WriteLine();
            htw.WriteBeginTag("table");
            if (this.BaseOptions.Gridlines) 
                htw.WriteAttribute("rules", "all");
            else
                htw.WriteAttribute("rules", "none");
            if (!string.IsNullOrEmpty(this.BaseOptions.GeneralCss))
                htw.WriteAttribute("class", this.BaseOptions.GeneralCss);
            htw.Write(HtmlTextWriter.TagRightChar);
        }

        private void WriteCellVertical(HtmlHelper htmlHelper, HtmlTextWriter htw, ScheduleTableCell cell)
        {
            string tag = cell.IsHeader ? "th" : "td";
            WriteCellOpeningTag(htw, tag, cell.ColumnSpan, cell.RowSpan, cell.CssClass,
                    cell.BorderTop, cell.BorderBottom, cell.BorderLeft, cell.BorderRight);
            if (cell.Data != null)
            {
                RenderItem(cell, htmlHelper, htw);
            }
            else
            {
                if (!string.IsNullOrEmpty(cell.Text))
                    htw.Write(cell.Text);
            }
            if (cell.IsHeader)
                htw.WriteEndTag("th");
            else
                htw.WriteEndTag("td");
        }

        private void WriteCellHorizontal(HtmlHelper htmlHelper, HtmlTextWriter htw, ScheduleTableCell cell)
        {
            string tag = cell.IsHeader ? "th" : "td";
            WriteCellOpeningTag(htw, tag, cell.RowSpan, cell.ColumnSpan, cell.CssClass,
                        cell.BorderLeft, cell.BorderRight, cell.BorderTop, cell.BorderBottom);
            if (!string.IsNullOrEmpty(cell.Text))
                htw.Write(cell.Text);
            if (cell.Data != null)
                RenderItem(cell, htmlHelper, htw);
            if (cell.IsHeader)
                htw.WriteEndTag("th");
            else
                htw.WriteEndTag("td");
        }

        protected virtual void RenderItem(ScheduleTableCell cell, HtmlHelper htmlHelper, HtmlTextWriter htw)
        {
            switch (cell.ItemType)
            {
                case ScheduleItemType.Item:
                case ScheduleItemType.AlternatingItem:
                    if (ItemDisplayExpression != null)
                        htw.Write((string)ItemDisplayExpression.Compile().Invoke((TItem)cell.Data));
                    else if (!string.IsNullOrEmpty(ItemPartialViewName))
                        htw.Write(htmlHelper.Partial(ItemPartialViewName, cell.Data));
                    else
                        throw new MvcScheduleException("Either ItemDisplayExpression or ItemPartialView should be provided");
                    break;
                case ScheduleItemType.TitleHeader:
                    if (TitleDisplayExpression != null)
                        htw.Write(TitleDisplayExpression.Compile().Invoke(cell.Data));
                    else
                        htw.Write(FormatDataValue(cell.Data, cell.ItemType));
                    break;
                case ScheduleItemType.RangeHeader:
                    if (DataRangeDisplayExpression != null)
                        htw.Write(DataRangeDisplayExpression.Compile().Invoke(cell.Data));
                    else
                        htw.Write(FormatDataValue(cell.Data, cell.ItemType));
                    break;
            }
        }

        private void WriteCellOpeningTag(HtmlTextWriter htw, string tag, int columnspan, int rowspan, string classname,
                                        bool? bordertop, bool? borderbottom, bool? borderleft, bool? borderright)
        {
            htw.WriteLine();
            htw.WriteBeginTag(tag);
            if (columnspan > 1)
                htw.WriteAttribute("colspan", columnspan.ToString());
            if (rowspan > 1)
                htw.WriteAttribute("rowspan", rowspan.ToString());
            if (!string.IsNullOrEmpty(classname))
                htw.WriteAttribute("class", classname);
            if (bordertop.HasValue || borderbottom.HasValue || borderleft.HasValue || borderright.HasValue)
            {
                htw.Write(" style=\"");
                if (bordertop.HasValue)
                    htw.WriteStyleAttribute("border-top", bordertop.Value ? "solid 1px" : "none");
                if (borderbottom.HasValue)
                    htw.WriteStyleAttribute("border-bottom", borderbottom.Value ? "solid 1px" : "none");
                if (borderleft.HasValue)
                    htw.WriteStyleAttribute("border-left", borderleft.Value ? "solid 1px" : "none");
                if (borderright.HasValue)
                    htw.WriteStyleAttribute("border-right", borderright.Value ? "solid 1px" : "none");
                htw.Write("\"");
            }
            htw.Write(HtmlTextWriter.TagRightChar);
        }

        #endregion

    }

}