﻿<%@ Page Language="C#" MasterPageFile="~/Default.master" Theme="Omega" Title="Articles | ASP.NET Application Error Handling" %>

<asp:Content ID="Content1" ContentPlaceHolderID="HeadContent" runat="server">
    
    <style type="text/css">
        #desktop img { border: 1px solid dimgrey; }
        code { font-size: 12px; }
    </style>

</asp:Content>

<asp:Content ID="Content2" ContentPlaceHolderID="TitleContent" runat="server">
    <div runat="server" id="BannerTitle" class="title">ASP.NET Application Error Handling</div>
    <div class="blurb">Tools and techniques for application development using ASP.NET and SQL Server</div>
</asp:Content>

<asp:Content ID="Content3" ContentPlaceHolderID="BodyContent" runat="server">
    
    <h1>Introduction</h1>
    
    <p>
        When an unhandled exception occurs in my application, I want my application to give the user a "graceful" response.
        Regardless of the error, I do not want the user to see an unfriendly technical error messages generated by IIS or ASP.NET.
        At the same time, I want to receive email notification for every unhandled exception.
    </p>

    <p>Theis article describes a simple and comprehensive solution to this problem.</p>
    
    <h1>The Problem</h1>
    
    <p>
        When I have no error handling configured for my application, my users might see any one of three different error pages,
        depending on the type of error.
    </p>
    <p>
        If a user requests a static resource that does not exist (for example, an HTML or JPG file),
        then the user sees the default HTTP error message generated by IIS:
    </p>
    
    <img src="default-iis-error-404.png" width="700px" />
    
    <p>
        If a user requests a dynamic resource that does not exist (for example, an ASPX file),
        then the user sees the default server error message generated by ASP.NET for HTTP 404 errors:
    </p>
    
    <img src="default-asp-error-404.png" width="700px" />
    
    <p>
        If an unhandled exception occurs in the application, 
        then the user sees the default server error message generated by ASP.NET for HTTP 500 errors:
    </p>

    <img src="default-asp-error-500.png" width="700px" />
    
    <p>
        ASP.NET web application developers sometimes call these
        the "Blue Screen of Death" (BSOD) and the "Yellow Screen of Death" (YSOD).
    </p>

    <h1>The Solution</h1>
    
    <p>
        When a user sees an error message in my application I want the error message to match the layout and style of 
        my application, and I want every error message to follow the same layout and style.
    </p>

    <h2>Step 1. Integrated Pipeline Mode</h2>

    <p>
        As a first step, I set my application to use an application pool that is configured for Integrated managed pipeline mode.
    </p>
    
    <img src="app-pool.png" />
    
    <p>
        Microsoft Internet Information System (IIS) version 6.0 (and previous versions) integrates ASP.NET as an ISAPI extension, alongside
        its own processing model for HTTP requests.  In effect, this gives two separate server pipelines: one for native components and one
        for managed components.  Managed components execute entirely within the ASP.NET ISAPI extension -- and only for requests specifically 
        mapped to ASP.NET.  IIS version 7.0 and above integrates these two pipelines so that services provided by both native and managed
        modules apply to all requests, regardless of the HTTP handler. 
    </p>
    
    <p>
        For more information on Integrated Pipeline mode, refer to this Microsoft article:<br/>
        <a href="http://www.iis.net/learn/application-frameworks/building-and-running-aspnet-applications/how-to-take-advantage-of-the-iis-integrated-pipeline">How to Take Advantage of the IIS 7.0 Integrated Pipeline</a>
    </p>
    
    <h2>Step 2. Application Configuration Settings</h2>
    
    <p>
        Next, I add error pages to my application for 404 and 500 error codes, and I update the application configuration file (web.config)
        with settings that instruct IIS to use my custom pages for these error codes.
    </p>

    <pre><code>
      &lt;system.webServer&gt;
        &lt;httpErrors errorMode="Custom" existingResponse="Replace"&gt;
          &lt;remove statusCode="404"/&gt;
          &lt;remove statusCode="500"/&gt;
          &lt;error statusCode="404" responseMode="ExecuteURL" path="/Pages/Public/Error404.aspx"/&gt;
          &lt;error statusCode="500" responseMode="ExecuteURL" path="/Pages/Public/Error500.aspx"/&gt;
        &lt;/httpErrors&gt;
      &lt;/system.webServer&gt;
    </code></pre>
    
    <p>
        Now, when a user requests a static resource that does not exist,
        the user sees the error message generated by my custom page:
    </p>
    
    <img src="custom-iis-error-404.png" width="700px" />
    
    <p>
        Similarly, if a user requests a dynamic resource that does not exist,
        then the user sees the error message generated by my custom page:
    </p>
    
    <img src="custom-asp-error-404.png" width="700px" />
    
    <p>
        And finally, if an unhandled exception occurs in the application, 
        then the user sees the error message generated by my custom page:
    </p>
    
    <img src="custom-asp-error-500.png" width="700px" />
    
    <h2>Step 3. Exception Details</h2>
    
    <p>
        The first part of my problem is now solved: the error messages seen by my users are rendered inside a 
        page that is consistent with the layout and style of my application, and the error messages themselves
        are consistent regardless of the underlying cause of the unhandled exception.
    </p>
    
    <p>
        However, in this configuration, when an unhandled exception occurs and IIS executes Error500.aspx, 
        the code behind this page has no details for the exception itself.  When an unhandled exception
        occurs, I need a crash report saved to the file system on the server and sent to me by email.  The
        crash report needs to include the exception details and a stack trace so that I can find and fix
        the cause of the error.
    </p>
    
    <p>
        Some articles suggest that I can identify the unhandled exception using Server.GetLastError, but
        I get a null value whenever I attempt this.
    </p>
    
    <pre><code>
      Exception ex = HttpContext.Current.Server.GetLastError(); // &lt;-- Returns null in Error500.aspx
    </code></pre>
    
    <p style="font-style:italic">
        Note: I have been unable to find a clear explanation for this in Microsoft's documentation. (Please drop me a note
        if you can direct me to the documentation that explains this behaviour.)
    </p>
    
    <p>
        I can solve this by adding an HTTP module with an event handler for application errors. For example, after I add
        this code to Global.asax, my custom error page can access the current cache for the exception details.
    </p>
    
    <pre><code>
    protected void Application_Error(object sender, EventArgs e)
    {
        Exception ex = HttpContext.Current.Server.GetLastError();
        CrashReport report = CrashReporter.CreateReport(ex, null);
        HttpContext.Current.Session[Settings.Names.CrashReport] = report;
    }
    </code></pre>
    
    <img src="custom-asp-error-500-improved.png" width="700px" />
    
    <p>
        It is important to note that if I add code at the end of my event handler to invoke Server.ClearError(),
        my custom error message is NOT displayed to the user (and neither is the default server error message).  In fact, if I invoke
        ClearError() here then the error message becomes a blank page, with no HTML in the outupt rendered to the browser.  
        Remember, the purpose of the event handler in this configuration is to store exception details in the current cache 
        (or in the session state) so that it is accessible to the Error500.aspx page.  The purpose is NOT to handle the exception 
        itself, and this is the reason the error is not cleared here.
    </p>

    <p style="font-style:italic">
        Note: Referring to my earlier point, if I have not cleared the error here, because it is required in order to ensure that my 
        custom error page is executed, then it is not obvious why the call to Server.GetLastError() in the custom error page
        returns a null value. If you have an explanation for this then please post a comment.
    </p>

    <h1>Improving the Solution</h1>
    
    <p>
        My solution needs to write a crash report to the file system (so we have a permanent record of the 
        event) and it needs to send an email notification (so we are immediately alerted to the event).

        At any given time my company is actively developing dozens of applications for various customers, so a reusable solution is
        important. 
    </p>
    <p>
        Code added to Global.asax is not easily reused across multiple applications, so I created an HTTP module (i.e., a class that
        inherits from System.Web.IHttpModule), which I can subsequently add to a library and then reference from many different applications.
    </p>
    <p>
        In order for this solution to work, I add the following settings to the system.webServer element in my
        web application configuration file (Web.config):
    </p>
    
    <pre><code>
    &lt;modules&gt;
        &lt;add name="ApplicationErrorModule" type="Demo.Classes.ApplicationErrorModule" /&gt;
    &lt;/modules&gt;
    </code></pre>
    
    <p>
        The code to wireup handling for an application error is simple:
    </p>
    
    <pre><code>
    public void Init(HttpApplication application)
    {
        application.Error += Application_Error;
    }
    
    private void Application_Error(Object sender, EventArgs e)
    {
        if (!Settings.Enabled)
            return;

        Exception ex = HttpContext.Current.Server.GetLastError();

        if (UnhandledExceptionOccurred != null)
            UnhandledExceptionOccurred(ex);

        ExceptionOccurred(ex);
    }
    </code></pre>
    
    <p>
        The code to process the exception itself is basically the same as the code I originally added to the global 
        application event handler, but here I also add code to save the crash report to the file system, and to
        send a copy to me by email.
    </p>
    
    <pre><code>
    private static void ExceptionOccurred(Exception ex)
    {
        // If the current request is itself an error page then we need to allow the exception to pass through.

        HttpRequest request = HttpContext.Current.Request;
        if (Regex.IsMatch(request.Url.AbsolutePath, ErrorPagePattern))
            return;

        // Otherwise, we should handle the exception here

        HttpResponse response = HttpContext.Current.Response;
        CrashReport report = new CrashReport(ex, null);

        // Save the crash report in the current cache so it is accessible to my custom error pages

        if (HttpContext.Current.Session != null)
            HttpContext.Current.Session[Settings.Names.CrashReport] = report;

        // Save the crash report on the file system
            
        String path = SaveCrashReport(report, request, null);

        // Send the crash report to the programmers

        SendEmail(report, path);
        
        // Write the crash report to the browser if there is no replacement defined for the HTTP response
            
        if (!ReplaceResponse)
        {
            HttpContext.Current.Server.ClearError();

            try
            {
                response.Clear();
                response.StatusCode = 500;
                response.StatusDescription = "Server Error";
                response.TrySkipIisCustomErrors = true;
                response.Write(report.Body);
                response.End();
            }
            catch { }
        }
    }
    </code></pre>
    
    <p>
        The last part of this function is especially important. If, for some reason, I forget to include the httpErrors
        section in my webServer configuration element, then I want the body of my crash report rendered to the browser
        and not the default Yellow Screen of Death (YSOD) that ASP.NET shows when a server error occurs.
    </p>

    <h1>Points of Interest</h1>
    
    <p>
        There are many good articles on the topic of ASP.NET application error handling, and there are many good products 
        helpful in the development of solutions.  Here are just a few references for more information:
    </p>
    
    <ul>
        <li><a href="http://www.iis.net/configreference/system.webserver/httperrors">IIS Configuration for HTTP Errors</a></li>
        <li><a href="http://www.codinghorror.com/blog/2007/03/creating-user-friendly-404-pages.html">Creating User-Friendly 404 Pages</a></li>
        <li><a href="http://www.4guysfromrolla.com/articles/090606-1.aspx">Gracefully Responding to Unhandled Exceptions</a></li>
        <li><a href="http://www.codeproject.com/Articles/10593/Error-Handling-in-ASP-NET">Error Handling in ASP.NET</a></li>
        <li><a href="http://code.google.com/p/elmah/">ASP.NET Error Logging Modules and Handlers (ELMAH)</a></li>
    </ul>

</asp:Content>
